Unit EtAsync;

{*****************************************************************************}
{* EtAsync V.1.06, 8/2 1993 Et-Soft                                          *}
{*                                                                           *}
{* Turbo Pascal unit with support for up to 8 serial ports.                  *}
{*****************************************************************************}

{$A-}                              {- Word alignment                         -}
{$B-}                              {- Complete boolean evaluation            -}
{$D-}                              {- Debug information                      -}
{$E-}                              {- Coprocessor emulation                  -}
{$F+}                              {- Force far calls                        -}
{$I-}                              {- I/O checkking                          -}
{$L-}                              {- Local debug symbols                    -}
{$N-}                              {- Coprocessor code generation            -}
{$O-}                              {- Overlayes allowed                      -}
{$R-}                              {- Range checking                         -}
{$S-}                              {- Stack checking                         -}
{$V-}                              {- Var-string checking                    -}
{$M 16384,0,655360}                {- Stack size, min heap, max heap         -}

Interface

Uses
  Dos;

{*****************************************************************************}

  {- Standard baudrates:                                                     -}
  {- 50, 75, 110, 134 (134.5), 150, 300, 600, 1200, 1800, 2000, 2400, 3600,  -}
  {- 4800, 7200, 9600, 19200, 38400, 57600, 115200                           -}


  Function OpenCOM            {- Open a COMport for communication            -}
    (Nr         : Byte;       {- Internal portnumber: 0-7                    -}
     Address    : Word;       {- Port address in hex: 000-3F8                -}
     IrqNum     : Byte;       {- Port Irq number: 0-15  (255 for no Irq)     -}
     Baudrate   : LongInt;    {- Baudrate: (see table)                       -}
     ParityBit  : Char;       {- Parity  : 'O','E' or 'N'                    -}
     Databits   : Byte;       {- Databits: 5-8                               -}
     Stopbits   : Byte;       {- Stopbits: 1-2                               -}
     BufferSize : Word;       {- Size of input buffer: 0-65535               -}
     Handshake  : Boolean)    {- True to use hardware handshake              -}
       : Boolean;             {- Returns true if ok                          -}
                              {- Note that the DTR signal is set if          -}
                              {- handshake is enabled (true).                -}

  Procedure CloseCOM          {- Close a open COMport                        -}
    (Nr : Byte);              {- Internal portnumber: 0-7                    -}

  Procedure ResetCOM          {- Reset a open COMport incl. buffer           -}
    (Nr : Byte);              {- Internal portnumber: 0-7                    -}

  Procedure COMSettings       {- Change settings for a open COMport          -}
    (Nr        : Byte;        {- Internal portnumber: 0-7                    -}
     Baudrate  : LongInt;     {- Baudrate: (see table)                       -}
     Paritybit : Char;        {- Parity  : 'O','E' or 'N'                    -}
     Databits  : Byte;        {- Databits: 5-8                               -}
     Stopbits  : Byte;        {- Stopbits: 1-2                               -}
     Handshake : Boolean);    {- True to use hardware handshake              -}
                              {- Note that the DTR signal is set if          -}
                              {- handshake is enabled (true).                -}

  Function COMAddress         {- Return the address for a COMport (BIOS)     -}
    (COMport : Byte)          {- COMport: 1-8                                -}
      : Word;                 {- Address found for COMport (0 if none)       -}

  Function WriteCOM           {- Writes a character to a port                -}
    (Nr : Byte;               {- Internal portnumber: 0-7                    -}
     Ch : Char)               {- Character to be written to port             -}
      : Boolean;              {- True if character send                      -}

  Function WriteCOMString     {- Writes a string to a port                   -}
    (Nr : Byte;               {- Internal portnumber: 0-7                    -}
     St : String)             {- String to be written to port                -}
      : Boolean;              {- True if string send                         -}

  Function CheckCOM           {- Check if any character is arrived           -}
    (Nr : Byte;               {- Internal portnumber: 0-7                    -}
     Var Ch : Char)           {- Character arrived                           -}
      : Boolean;              {- Returns true and character if any           -}

Function COMCheck(Nr : Byte; Var Ch : Char) : Boolean;
                              {- Checker frste char uden at slette (MR)}

  Function COMError           {- Returns status of the last operation        -}
      : Integer;              {- 0 = Ok                                      -}
                              {- 1 = Not enough memory                       -}
                              {- 2 = Port not open                           -}
                              {- 3 = Port already used once                  -}
                              {- 4 = Selected Irq already used once          -}
                              {- 5 = Invalid port                            -}
                              {- 6 = Timeout                                 -}
                              {- 7 = Port failed loopback test               -}
                              {- 8 = Port failed IRQ test                    -}

  Function COMErrorStr        {- Returns a string repesentation of COMError  -}
    (ErrCode : Integer)       {- The error number                            -}
      : String;               {- The string                                  -}

  Function TestCOM            {- Performs a loopback and IRQ test on a port  -}
    (Nr : Byte)               {- Internal port number: 0-7                   -}
      : Boolean;              {- True if port test ok                        -}
                              {- Note: This test is performed during OpenCOM -}
                              {- if enabled (TestCOM is by default disabled. -}

  Procedure EnableTestCOM;    {- Enable TestCOM during Openport              -}

  Procedure DisableTestCOM;   {- Disable TestCOM during Openport (Default)   -}

  Function COMUsed            {- Check whether or not a port is open         -}
    (Nr : Byte)               {- Internal port number: 0-7                   -}
      : Boolean;              {- True if port is open and in use             -}
                              {- Note: This rutine can not test whether or   -}
                              {- not a COMport is used by another application-}

  Function IrqUsed            {- Check whether or not an Irq is used         -}
    (IrqNum : Byte)           {- Irq number: 0-15                            -}
      : Boolean;              {- True if Irq is used                         -}
                              {- Note: This rutine can not test whether or   -}
                              {- not an IRQ is used by another application   -}

  Function IrqInUse           {- Test IRQ in use on the PIC                  -}
    (IrqNum : Byte)           {- Irq number: 0-15                            -}
      : Boolean;              {- True if Irq is used                         -}

  Procedure SetIrqPriority    {- Set the Irq priority level on the PIC       -}
    (IrqNum : Byte);          {- Irq number: 0-15                            -}
                              {- The IrqNum specified will get the highest   -}
                              {- priority, the following Irq number will then-}
                              {- have the next highest priotity and so on    -}

  Procedure ClearBuffer       {- Clear the input buffer for a open port      -}
    (Nr : Byte);              {- Internal port number: 0-7                   -}

  Procedure SetDTR            {- Set the "Data Terminal Ready" signal        -}
    (Nr : Byte;               {- Internal port number: 0-7                   -}
     On : Boolean);           {- True to set (on), False to clear (off)      -}
                              {- Note that the DTR signal is set in OpenCOM, -}
                              {- if handshake is enabled.                    -}

  Procedure SetRTS            {- Set the "Request To Send" signal            -}
    (Nr : Byte;               {- Internal port number: 0-7                   -}
     On : Boolean);           {- True to set (on), False to clear (off)      -}
                              {- Note that the RTS signal is used internally -}
                              {- when the hardware handshake is enabled.     -}

  Function DTRstate           {- Return state of "Data Terminal Ready"       -}
    (Nr : Byte)               {- Internal port number: 0-7                   -}
    : Boolean;                {- True if low (on), false if high (off)       -}

  Function RTSstate           {- Return state of "Request to Send"           -}
    (Nr : Byte)               {- Internal port number: 0-7                   -}
    : Boolean;                {- True if low (on), false if high (off)       -}

  Function CTSstate           {- Return state of the "Clear To Send" input   -}
    (Nr : Byte)               {- Internal port number: 0-7                   -}
    : Boolean;                {- True if low (on), false if high (off)       -}

  Function DSRstate           {- Return state of the "Data Set Ready" input  -}
    (Nr : Byte)               {- Internal port number: 0-7                   -}
    : Boolean;                {- True if low (on), false if high (off)       -}

  Function RIstate            {- Return state of the "Ring Indicator" input  -}
    (Nr : Byte)               {- Internal port number: 0-7                   -}
    : Boolean;                {- True if low (on), false if high (off)       -}

  Function RLSDstate          {- State of "Recv. Line Signal Detect" input   -}
    (Nr : Byte)               {- Internal port number: 0-7                   -}
    : Boolean;                {- True if low (on), false if high (off)       -}

  Function ModemStatus        {- Status from CTS, DSR, RI and RLSD in 1 byte -}
    (Nr : Byte)               {- Internal port number: 0-7                   -}
    : Byte;                   {- Returns a byte, with information on all     -}
                              {- status inputs. This modem status byte is    -}
                              {- split into the different "state" functions, -}
                              {- declared above, for easy use.               -}
                              {- Bit 0 = Delta CTS (Clear To Send)           -}
                              {- Bit 1 = Delta DSR (Data Set Ready)          -}
                              {- Bit 2 = Trailing Edge RI (Ring Indicator)   -}
                              {- Bit 3 = Delta RLSD (Recv. Line Signal Detect }
                              {- Bit 4 = CTS                                 -}
                              {- Bit 4 = DSR                                 -}
                              {- Bit 4 = RI                                  -}
                              {- Bit 4 = RLSD                                -}
                              {- The delta bits are set when the status of   -}
                              {- associated signal has changed state since   -}
                              {- last modem status call. The trailing edge   -}
                              {- RI is set when the RI signal has changed    -}
                              {- from on (logic 1) to off (logic 0) since    -}
                              {- last call to modem status. The CTS,DSR,RI & -}
                              {- RLSD bits are set when the apropriate signal-}
                              {- is on (logic 0).                            -}
                              {- Please note that the modem status is used   -}
                              {- internal when hardware handshake is enabled -}
                              {- and therefore the first 4 bits may be reset -}
                              {- many times.                                 -}

{*****************************************************************************}

Implementation

Type
  Buffer = Array[1..65535] Of Byte;  {- Dummy type for Interrupt buffer      -}

  PortRec = Record                   {- Portdata type                        -}
    InUse   : Boolean;               {- True if port is used                 -}
    Addr    : Word;                  {- Selected address                     -}
    Irq     : Byte;                  {- Selected Irq number                  -}
    PIC     : Word;                  {- Address for PIC in use               -}
    OldIrq  : Byte;                  {- Status of Irq before InitCOM         -}
    HShake  : Boolean;               {- Hardware handshake on/off            -}

    Buf     : ^Buffer;               {- Pointer to allocated buffer          -}
    BufSize : Word;                  {- Size of allocated buffer             -}
    OldVec  : Pointer;               {- Saved old interrupt vector           -}

    Baud    : LongInt;               {- Selected baudrate                    -}
    Parity  : Char;                  {- Selected parity                      -}
    Databit : Byte;                  {- Selected number of databits          -}
    Stopbit : Byte;                  {- Selected number of stopbits          -}

    InPtr   : Word;                  {- Pointer to buffer input index        -}
    OutPtr  : Word;                  {- Pointer to buffer output index       -}

    Reg0    : Byte;                  {- Saved UART register 0                -}
    Reg1    : Array[1..2] Of Byte;   {- Saved UART register 1's              -}
    Reg2    : Byte;                  {- Saved UART register 2                -}
    Reg3    : Byte;                  {- Saved UART register 3                -}
    Reg4    : Byte;                  {- Saved UART register 4                -}
    Reg6    : Byte;                  {- Saved UART register 6                -}
  End;

Var
  COMResult   : Integer;                    {- Last Error (Call COMError)    -}
  ExitChainP  : Pointer;                    {- Saved Exitproc pointer        -}
  PICStatus   : Array[0..1] Of Byte;        {- PIC status save variable      -}
  Ports       : Array[0..7] Of PortRec;     {- The 8 ports supported         -}

Const
  PICA : Array[0..1] Of Word = ($20,$A0);   {- Default PIC addresses         -}
  EOI  = $20;                               {- PIC control byte              -}
  TestCOMEnabled : Boolean = False;         {- Test port during OpenCOM      -}

Procedure DisableInterrupts;                {- Disable interrupt             -}
  Begin
    Inline($FA);                            {- CLI (Clear Interruptflag)     -}
  End;

Procedure EnableInterrupts;                 {- Enable interrupts             -}
  Begin
    Inline($FB);                            {- STI (Set interrupt flag)      -}
  End;

Procedure Port0Int; Interrupt;                   {- Interrupt rutine port 0  -}

  Begin
    With Ports[0] Do
      Begin
        Buf^[InPtr] := Port[Addr];                {- Read data from port     -}
        Inc(InPtr);                               {- Count one step forward.. }
        If InPtr > BufSize Then InPtr := 1;       {  .. in buffer            -}
        Port[PIC] := EOI;                         {- Send EOI to PIC         -}
      End;
  End;

Procedure Port1Int; Interrupt;                    {- Interrupt rutine port 1 -}

  Begin
    With Ports[1] Do
      Begin
        Buf^[InPtr] := Port[Addr];                {- Read data from port     -}
        Inc(InPtr);                               {- Count one step forward.. }
        If InPtr > BufSize Then InPtr := 1;       {  .. in buffer            -}
        Port[PIC] := EOI;                         {- Send EOI to PIC         -}
      End;
  End;

Procedure Port2Int; Interrupt;                    {- Interrupt rutine port 2 -}

  Begin
    With Ports[2] Do
      Begin
        Buf^[InPtr] := Port[Addr];                {- Read data from port     -}
        Inc(InPtr);                               {- Count one step forward.. }
        If InPtr > BufSize Then InPtr := 1;       {  .. in buffer            -}
        Port[PIC] := EOI;                         {- Send EOI to PIC         -}
      End;
  End;

Procedure Port3Int; Interrupt;                    {- Interrupt rutine port 3 -}

  Begin
    With Ports[3] Do
      Begin
        Buf^[InPtr] := Port[Addr];                {- Read data from port     -}
        Inc(InPtr);                               {- Count one step forward.. }
        If InPtr > BufSize Then InPtr := 1;       {  .. in buffer            -}
        Port[PIC] := EOI;                         {- Send EOI to PIC         -}
      End;
  End;

Procedure Port4Int; Interrupt;                    {- Interrupt rutine port 4 -}

  Begin
    With Ports[4] Do
      Begin
        Buf^[InPtr] := Port[Addr];                {- Read data from port     -}
        Inc(InPtr);                               {- Count one step forward.. }
        If InPtr > BufSize Then InPtr := 1;       {  .. in buffer            -}
        Port[PIC] := EOI;                         {- Send EOI to PIC         -}
      End;
  End;

Procedure Port5Int; Interrupt;                    {- Interrupt rutine port 5 -}

  Begin
    With Ports[5] Do
      Begin
        Buf^[InPtr] := Port[Addr];                {- Read data from port     -}
        Inc(InPtr);                               {- Count one step forward.. }
        If InPtr > BufSize Then InPtr := 1;       {  .. in buffer            -}
        Port[PIC] := EOI;                         {- Send EOI to PIC         -}
      End;
  End;

Procedure Port6Int; Interrupt;                    {- Interrupt rutine port 6 -}

  Begin
    With Ports[6] Do
      Begin
        Buf^[InPtr] := Port[Addr];                {- Read data from port     -}
        Inc(InPtr);                               {- Count one step forward.. }
        If InPtr > BufSize Then InPtr := 1;       {  .. in buffer            -}
        Port[PIC] := EOI;                         {- Send EOI to PIC         -}
      End;
  End;

Procedure Port7Int; Interrupt;                    {- Interrupt rutine port 7 -}

  Begin
    With Ports[7] Do
      Begin
        Buf^[InPtr] := Port[Addr];                {- Read data from port     -}
        Inc(InPtr);                               {- Count one step forward.. }
        If InPtr > BufSize Then InPtr := 1;       {  .. in buffer            -}
        Port[PIC] := EOI;                         {- Send EOI to PIC         -}
      End;
  End;

Procedure InitPort(Nr : Byte; SaveStatus : Boolean);   {- Port initialize    -}

Var
  Divider  : Word;                                {- Baudrate divider number -}
  CtrlBits : Byte;                                {- UART control byte       -}

  Begin
    With Ports[Nr] Do
      Begin
        Divider := 115200 Div Baud;                 {- Calc baudrate divider -}

        CtrlBits := DataBit - 5;                    {- Insert databits       -}

        If Parity <> 'N' Then
          Begin
            CtrlBits := CtrlBits Or $08;            {- Insert parity enable  -}
            If Parity = 'E' Then                    {- Enable even parity    -}
              CtrlBits := CtrlBits Or $10;
          End;

        If Stopbit = 2 Then
          CtrlBits := CtrlBits Or $04;              {- Insert stopbits       -}

        If SaveStatus Then Reg3 := Port[Addr + $03];    {- Save register 3   -}
        Port[Addr + $03] := $80;                        {- Baudrate change   -}

        If SaveStatus Then Reg0 := Port[Addr + $00];    {- Save Lo Baud      -}
        Port[Addr + $00] := Lo(Divider);                {- Set Lo Baud       -}

        If SaveStatus Then Reg1[2] := Port[Addr + $01]; {- Save Hi Baud      -}
        Port[Addr + $01] := Hi(Divider);                {- Set Hi Baud       -}

        Port[Addr + $03] := CtrlBits;                   {- Set control reg.  -}
        If SaveStatus Then Reg6 := Port[Addr + $06];    {- Save register 6   -}
      End;
  End;

Function IrqUsed(IrqNum : Byte) : Boolean;

Var
  Count : Byte;
  Found : Boolean;

  Begin
    Found := False;                                 {- Irq not found         -}
    Count := 0;                                     {- Start with port 0     -}

    While (Count <= 7) And Not Found Do             {- Count the 8 ports     -}
      With Ports[Count] Do
        Begin
          If InUse Then
            Found := IrqNum = Irq;                  {- Check Irq match       -}
          Inc(Count);                               {- Next port             -}
        End;

    IrqUsed := Found;                               {- Return Irq found      -}
  End;

Procedure EnableTestCOM;
  Begin
    TestCOMEnabled := True;
  End;

Procedure DisableTestCOM;
  Begin
    TestCOMEnabled := False;
  End;

Function TestCOM(Nr : Byte) : Boolean;

Var
  OldReg0   : Byte;
  OldReg1   : Byte;
  OldReg4   : Byte;
  OldReg5   : Byte;
  OldReg6   : Byte;
  OldInPtr  : Word;
  OldOutPtr : Word;
  TimeOut   : LongInt;

  Begin
    TestCOM := False;

    With Ports[Nr] Do
      Begin
        If InUse Then
          Begin
            OldInPtr  := InPtr;
            OldOutPtr := OutPtr;
            OldReg1 := Port[Addr + $01];
            OldReg4 := Port[Addr + $04];
            OldReg5 := Port[Addr + $05];
            OldReg6 := Port[Addr + $06];

            Port[Addr + $05] := $00;
            Port[Addr + $04] := Port[Addr + $04] Or $10;

            OldReg0 := Port[Addr + $00];
            OutPtr  := InPtr;

            TimeOut := 100000;
            Port[Addr + $00] := OldReg0;

            While (Port[Addr + $05] And $01 = $00) And (TimeOut <> 0) Do
              Dec(TimeOut);

            If TimeOut <> 0 Then
              Begin
                If Port[Addr + $00] = OldReg0 Then
                  Begin
                    If IRQ In [0..15] Then
                      Begin
                        TimeOut := 100000;
                        OutPtr  := InPtr;

                        Port[Addr + $01] := $08;
                        Port[Addr + $04] := $08;
                        Port[Addr + $06] := Port[Addr + $06] Or $01;

                        While (InPtr = OutPtr) And (TimeOut <> 0) Do
                          Dec(TimeOut);

                        Port[Addr + $01] := OldReg1;

                        If (InPtr <> OutPtr) Then TestCOM := True
                        Else COMResult := 8;
                      End
                    Else
                      TestCOM := True;
                  End
                Else
                  COMResult := 7;            {- Loopback test failed -}
              End
            Else
              COMResult := 6;                {- Timeout -}

            Port[Addr + $04] := OldReg4;
            Port[Addr + $05] := OldReg5;
            Port[Addr + $06] := OldReg6;

            For TimeOut := 1 to MaxInt Do;
            If Port[Addr + $00] = 0 Then;

            InPtr  := OldInPtr;
            OutPtr := OldOutPtr;
          End
        Else
          COMResult := 2;                    {- Port not open -}
      End;
  End;

Procedure CloseCOM(Nr : Byte);

  Begin
    With Ports[Nr] Do
      Begin
        If InUse Then
          Begin
            InUse := False;

            If Irq <> 255 Then                          {- If Interrupt used -}
              Begin
                FreeMem(Buf,BufSize);                   {- Deallocate buffer -}
                DisableInterrupts;
                Port[$21] := Port[$21] Or ($01 Shl Irq) And OldIrq; {-restore-}
                Port[Addr + $04] := Reg4;               {- Disable UART OUT2 -}
                Port[Addr + $01] := Reg1[1];            {- Disable UART Int. -}
                SetIntVec($08+Irq,OldVec);              {- Restore Int.Vector-}
                EnableInterrupts;
              End;
            Port[Addr + $03] := $80;                    {- UART Baud set     -}
            Port[Addr + $00] := Reg0;                   {- Reset Lo Baud     -}
            Port[Addr + $01] := Reg1[2];                {- Reset Hi Baud     -}
            Port[Addr + $03] := Reg3;                   {- Restore UART ctrl.-}
            Port[Addr + $06] := Reg6;                   {- Restore UART reg6 -}
          End
        Else
          COMResult := 2;                               {- Port not in use   -}
      End;
  End;

Function OpenCOM
 (Nr : Byte; Address  : Word; IrqNum : Byte; Baudrate : LongInt;
  ParityBit : Char; Databits, Stopbits : Byte; BufferSize : Word;
  HandShake : Boolean) : Boolean;

Var
  IntVec : Pointer;
  OldErr : Integer;

  Begin
    OpenCOM := False;

    If (IrqNum = 255) Or
    ((IrqNum In [0..15]) And (MaxAvail >= LongInt(BufferSize)) And Not
IrqUsed(IrqNum)) Then
      With Ports[Nr] Do
        Begin
          If Not InUse And (Address <= $3F8) Then
            Begin
              InUse   := True;                    {- Port now in use         -}

              Addr    := Address;                 {- Save parameters         -}
              Irq     := IrqNum;
              HShake  := HandShake;
              BufSize := BufferSize;
              Baud    := Baudrate;

              If Ord(Paritybit) In [97..122] Then
                Parity := Chr(Ord(Paritybit)-32)  {- Upcase character        -}
              Else
                Parity := ParityBit;              {- Save character as is    -}

              Databit := Databits;
              Stopbit := Stopbits;
              PIC     := PICA[Irq And $08 Shr 3]; {- PIC address             -}
              InPtr   := 1;                       {- Reset InputPointer      -}
              OutPtr  := 1;                       {- Reset OutputPointer     -}

              If (Irq In [0..15]) And (BufSize > 0) Then
                Begin
                  GetMem(Buf,BufSize);            {- Allocate buffer         -}
                  GetIntVec($08+Irq,OldVec);      {- Save Interrupt vector   -}

                  Case Nr Of                      {- Find the interrupt proc.-}
                    0 : IntVec := @Port0Int;
                    1 : IntVec := @Port1Int;
                    2 : IntVec := @Port2Int;
                    3 : IntVec := @Port3Int;
                    4 : IntVec := @Port4Int;
                    5 : IntVec := @Port5Int;
                    6 : IntVec := @Port6Int;
                    7 : IntVec := @Port7Int;
                  End;

                  Reg1[1] := Port[Addr + $01];    {- Save register 1 -}
                  Reg4    := Port[Addr + $04];    {- Save register 4 -}

                  OldIrq  := Port[PIC+1] Or Not ($01 Shl (Irq and $0007));
                                                  {- Save Irq status -}
                  DisableInterrupts;              {- Disable interrupts -}
                  SetIntVec($08+Irq,IntVec);      {- Set the interrupt vector -}
                  Port[Addr + $04] := $08;        {- Enable OUT2 on port -}
                  Port[Addr + $01] := $01;        {- Set port data avail.int. -}
                  Port[PIC+1] := Port[PIC+1] And Not ($01 Shl (Irq And $07));
                                                  {- Enable Irq -}
                  EnableInterrupts;               {- Enable interrupts again -}
                End;

              InitPort(Nr,True);                  {- Initialize port -}
              COMResult := 0;
              OpenCOM   := True;

              If TestCOMEnabled And Not TestCOM(Nr) Then
                Begin
                  OldErr    := COMResult;
                  CloseCOM(Nr);
                  COMResult := OldErr;
                  OpenCOM   := False;
                End;
              If Port[Addr + $00] = 0 Then;  {- Remove any pending character -}
              If Port[Addr + $05] = 0 Then;  {- Reset line status register   -}
              InPtr := OutPtr;               {- Reset the input buffer       -}
              If HShake Then SetDTR(Nr,True);
            End
          Else If InUse Then
            COMResult := 3                        {- Port already in use -}
          Else If (Address > $3F8) Then
            COMResult := 5;                       {- Invalid port address -}
        End
    Else If (MaxAvail >= BufferSize) Then         {- Not enough memory -}
      COMResult := 1
    Else If IrqUsed(IrqNum) Then                  {- Irq already used -}
      COMResult := 4;
  End;

Procedure ResetCOM(Nr : Byte);

  Begin
    With Ports[Nr] Do
      Begin
        If InUse Then                        {- Is port defined ?            -}
          Begin
            InPtr  := 1;                     {- Reset buffer pointers        -}
            OutPtr := 1;
            InitPort(Nr,False);              {- Reinitialize the port        -}

            If Port[Addr + $00] = 0 Then;    {- Remove any pending character -}
            If Port[Addr + $05] = 0 Then;    {- Reset line status register   -}
          End
        Else
          COMResult := 2;                    {- Port not open                -}
      End;
  End;

Procedure COMSettings(Nr : Byte; Baudrate : LongInt; ParityBit : Char;
  Databits, Stopbits : Byte; HandShake : Boolean);

  Begin
    With Ports[Nr] Do
      Begin
        If InUse Then                                {- Is port in use       -}
          Begin
            Baud    := Baudrate;                     {- Save parameters      -}
            Databit := Databits;
            Stopbit := Stopbits;
            HShake  := HandShake;

            If Ord(Paritybit) In [97..122] Then
              Parity := Chr(Ord(Paritybit)-32)       {- Upcase character     -}
            Else
              Parity := ParityBit;                   {- Save character as is -}

            InitPort(Nr,False);                      {- ReInit port          -}
          End
        Else
          COMResult := 2;                            {- Port not in use      -}
      End;
  End;

Function COMAddress(COMport : Byte) : Word;

  Begin
    If COMPort In [1..8] Then
      COMAddress := MemW[$40:(Pred(COMPort) Shl 1)]       {- BIOS data table -}
    Else
      COMResult := 5;                                     {- Invalid port    -}
  End;

Function WriteCOM(Nr : Byte; Ch : Char) : Boolean;

Var
  Count : Integer;

  Begin
    WriteCom := True;

    With Ports[Nr] Do
      If InUse Then
        Begin
          While Port[Addr + $05] And $20 = $00 Do;   {- Wait until char send -}

          If Not HShake Then
            Port[Addr] := Ord(Ch)                    {- Send char to port    -}
          Else
            Begin
              Port[Addr + $04] := Port[Addr + $04] Or $02;        {- Set RTS -}
              Port[Addr + $04] := $0B;
              Count := MaxInt;

              While (Port[Addr + $06] And $10 = 0) And (Count <> 0) Do
                Dec(Count);                          {- Wait for CTS         -}

              If Count <> 0 Then                     {- If not timeout       -}
                Port[Addr] := Ord(Ch)                {- Send char to port    -}
              Else
                Begin
                  COMResult := 6;                    {- Timeout error        -}
                  WriteCom  := False;
                End;
            End;
        End
      Else
        Begin
          COMResult := 2;                            {- Port not in use      -}
          WriteCom  := False;
        End;
  End;

Function WriteCOMString(Nr : Byte; St : String) : Boolean;

Var
  Ok : Boolean;
  Count : Byte;

  Begin
    If Length(St) > 0 Then                           {- Any chars to send ?  -}
      Begin
        Ok    := True;
        Count := 1;
        While (Count <= Length(St)) And Ok Do        {- Count chars          -}
          Begin
            Ok := WriteCOM(Nr,St[Count]);            {- Send char            -}
            Inc(Count);                              {- Next character       -}
          End;
        WriteCOMString := Ok;                        {- Return status        -}
      End;
  End;

Function CheckCOM(Nr : Byte; Var Ch : Char) : Boolean;

  Begin
    With Ports[Nr] Do
      Begin
        If InPtr <> OutPtr Then                      {- Any char in buffer ? -}
          Begin
            Ch := Chr(Buf^[OutPtr]);                 {- Get char from buffer -}
            Inc(OutPtr);                             {- Count outpointer up  -}
            If OutPtr > BufSize Then OutPtr := 1;
            CheckCOM := True;
          End
        Else
          CheckCOM := False;                         {- No char in buffer    -}
      End;
  End;


Function COMCheck(Nr : Byte; Var Ch : Char) : Boolean;

  Begin
    With Ports[Nr] Do
      Begin
        If InPtr <> OutPtr Then                      {- Any char in buffer ? -}
          Begin
            Ch := Chr(Buf^[OutPtr]);                 {- Get char from buffer -}
            {Inc(OutPtr);}                             {- Count outpointer up -}
            If OutPtr > BufSize Then OutPtr := 1;
            COMCheck := True;
          End
        Else
          COMCheck := False;                         {- No char in buffer    -}
      End;
  End;

Function COMError : Integer;

  Begin
    COMError := COMResult;                           {- Return last error    -}
    COMResult := 0;
  End;

Function COMErrorStr(ErrCode : Integer) : String;

Var
  S : String;

  Begin
    Case ErrCode Of
      0 : S := 'Ok';
      1 : S := 'Not enough memory';
      2 : S := 'Port not open';
      3 : S := 'Port already used once';
      4 : S := 'Selected Irq already used once';
      5 : S := 'Invalid port';
      6 : S := 'Timeout';
      7 : S := 'Port failed loopback test';
      8 : S := 'Port failed IRQ test';
    End;
    COMErrorStr := S;
  End;

Function COMUsed(Nr : Byte) : Boolean;

  Begin
    COMUsed := Ports[Nr].InUse;                      {- Return used status   -}
  End;

Function IrqInUse(IrqNum : Byte) : Boolean;

Var
  IrqOn : Byte;
  Mask  : Byte;
  PIC : Word;

  Begin
    IrqInUse := False;

    If IrqNum In [0..15] Then
      Begin
        PIC      := PICA[IrqNum And $80 Shr 3];
        IrqNum   := IrqNum And $0007;
        IrqOn    := Port[PIC+1];
        Mask     := ($01 Shl IrqNum);
        IrqInUse := IrqOn Or Not Mask = Not Mask;
      End;
  End;

Procedure SetIrqPriority(IrqNum : Byte);

Var
  PIC : Word;

  Begin
    If IrqNum In [0..15] Then
      Begin
        PIC    := PICA[IrqNum And $80 Shr 3];
        IrqNum := IrqNum And $0007;
        If IrqNum > 0 Then Dec(IrqNum)
        Else IrqNum := 7;
        DisableInterrupts;
        Port[PIC] := $C0 + IrqNum;
        EnableInterrupts;
      End;
  End;

Procedure ClearBuffer(Nr : Byte);

  Begin
    With Ports[Nr] Do If InUse And (BufSize > 0) Then OutPtr := InPtr;
  End;

Procedure SetDTR(Nr : Byte; On : Boolean);

  Begin
    With Ports[Nr] Do If InUse Then
      Begin
        If On Then Port[Addr+$04] := Port[Addr+$04] Or $01      {- Set DTR   -}
        Else Port[Addr+$04] := Port[Addr+$04] And $FE;          {- Clear DTR -}
      End;
  End;

Procedure SetRTS(Nr : Byte; On : Boolean);

  Begin
    With Ports[Nr] Do If InUse Then
      Begin
        If On Then Port[Addr+$04] := Port[Addr+$04] Or $02      {- Set RTS   -}
        Else Port[Addr+$04] := Port[Addr+$04] And $FD;          {- Clear RTS -}
      End;
  End;

Function DTRstate(Nr : Byte) : Boolean;
  Begin
    DTRstate := False;
    With Ports[Nr] Do If InUse Then DTRstate := Port[Addr+$04] And $01 = $01;
  End;

Function RTSstate(Nr : Byte) : Boolean;
  Begin
    RTSstate := False;
    With Ports[Nr] Do If InUse Then RTSstate := Port[Addr+$04] And $02 = $02;
  End;

Function CTSstate(Nr : Byte) : Boolean;
  Begin
    CTSstate := False;
    With Ports[Nr] Do If InUse Then CTSstate := Port[Addr+$06] And $10 = $10;
  End;

Function DSRstate(Nr : Byte) : Boolean;
  Begin
    DSRstate := False;
    With Ports[Nr] Do If InUse Then DSRstate := Port[Addr+$06] And $20 = $20;
  End;

Function RIstate(Nr : Byte) : Boolean;
  Begin
    RIstate := False;
    With Ports[Nr] Do If InUse Then RIstate := Port[Addr+$06] And $40 = $40;
  End;

Function RLSDstate(Nr : Byte) : Boolean;
  Begin
    RLSDstate := False;
    With Ports[Nr] Do If InUse Then RLSDstate := Port[Addr+$06] And $80 = $80;
  End;

Function ModemStatus(Nr : Byte) : Byte;
  Begin
    ModemStatus := $00;
    With Ports[Nr] Do If InUse Then ModemStatus := Port[Addr+$06];
  End;

Procedure DeInit;

Var
  Count : Byte;

  Begin
    For Count := 0 to 7 Do CloseCOM(Count);         {- Close open ports     -}

    DisableInterrupts;
    For Count := 0 to 1 Do Begin
      Port[PICA[Count]+1] := PICStatus[Count];      {- PIC status restore   -}
      Port[PICA[Count]]   := $C7;                   {- PIC Priority reset   -}
    End;
    EnableInterrupts;

    ExitProc := ExitChainP;                         {- Restore ExitProc     -}
  End;

Procedure Init;

Var
  Count : Byte;

  Begin
    COMResult  := 0;
    ExitChainP := ExitProc;                          {- Save ExitProc        -}
    ExitProc   := @DeInit;                           {- Set ExitProc         -}

    For Count := 0 To 7 Do
      Ports[Count].InUse := False;                   {- No ports open        -}

    For Count := 0 to 1 Do
      PICStatus[Count] := Port[PICA[Count]+1];       {- Save PIC status      -}
  End;

{*****************************************************************************}

Begin
  Init;
End.
=== Cut ===

Hi Ricky!

If this code dont work, you can mail you boss phone-number (the bbs your calling
to get you mail) and i'll send the code to you.

Regards,
Morten

--- GEcho 1.00
 * Origin:  ----> Point #52 p BABU BBS  45 65 13 53 <----  (2:236/447.52)
