/*
 * minigl_main.c
 *
 * miniGL for the Palm Computing platform
 * Michael Sherman <msherman@dsbox.com>
 * (c) 2000 Digital Sandbox, Inc.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/** For experimental z-buffer */
//#define ZBUFFER 1

/** For frames per second printout on screen (top right corner) */
#define PRINT_FPS 1

/** OpenGL spec says mins are 32 for modv and 2 for proj */
#define MAX_MAT_STACK_MODV 32
#define MAX_MAT_STACK_PROJ 2

#include "miniGL.h"
//#include "fastmath.h"
#include "fastgraph.h"

#ifdef FIXED_MATH
typedef long fixed;
#else
typedef float fixed;
#endif

#ifdef FIXED_MATH
#define ITF(x)  ((fixed)(x) << 16)
#define FTI(x)  ((x) >> 16)
#else
#define ITF(x) ((float)(x))
#define FTI(x) ((int)(x))
#endif

fixed DTF(float f) {
#ifdef FIXED_MATH
	const unsigned long i = *(unsigned long*)&f + ((unsigned long)16 << 23);
	return (fixed)*(const float*)&i;
#else
	return f;
#endif
}

float FTD(fixed fix) {
#ifdef FIXED_MATH
	const float f = (float)fix;
	const unsigned long i = *(const unsigned long*)&f -
		((unsigned long)16 << 23);

	return *(const float*)&i;
#else
	return fix;
#endif
}

static fixed modv_matrix[MAX_MAT_STACK_MODV][16]; /** geometry transform matrix */
static char modv_level = 0;
static fixed proj_matrix[MAX_MAT_STACK_PROJ][16]; /** perspective transform matrix */
static char proj_level = 0;
//static fixed* cur_matrix = modv_matrix[0];
static float cur_matrix[16];
static float per_matrix[16];
static fixed scr_matrix[16]; /** screen coord transform matrix */
static GLfloat cur_color[4];
static GLfloat cur_normal[4];
static GLfloat vertices_color[MAX_VERTICES][4];
static GLfloat vertices_normal[MAX_VERTICES][4];
static GLfloat vertices[MAX_VERTICES][4];
static GLfloat scr_vertices[MAX_VERTICES][3];
static int num_vertices=0, screen_width, screen_height, culling=0,
	screen_startx, screen_starty, two_created=0, lighting=0, wireframe=0;
static GLenum cur_mode, cur_shade_model;
static GLenum matrix_mode = GL_MODELVIEW;
static WinHandle one, two;     /** windows for double-buffering */
static Light lights[8];

/** Beginning z-buffer implementation */
static GLubyte *zbuffer; /** 8-bit depth buffer */

static fixed fixmul(fixed a, fixed b) {
#ifdef FIXED_MATH
	return ((a >> 6)*(b >> 6)) >> 4;
#else
	return a * b;
#endif
}

static fixed fixdiv(fixed a, fixed b) {
#ifdef FIXED_MATH
	return ((a << 7)/(b >> 9));
#else
	return a / b;
#endif
}

/*
 * Mutiply a 4x4 matrix by a 4x1 vector to result in a 4x1 vector.
 *
 * m - input 4x4 matrix
 * p - input 4x1 vector
 * ret - output 4x1 matrix
 */
static void MatrixMultVector(const fixed *m, const GLfloat *p, GLfloat *ret) {

#ifdef FIXED_MATH
	const fixed p0 = DTF(p[0]);
	const fixed p1 = DTF(p[1]);
	const fixed p2 = DTF(p[2]);
	const fixed p3 = DTF(p[3]);

	ret[0] = FTD(fixmul(m[0], p0) + fixmul(m[4], p1) +
		fixmul(m[8], p2) + fixmul(m[12], p3));
	ret[1] = FTD(fixmul(m[1], p0) + fixmul(m[5], p1) +
		fixmul(m[9], p2) + fixmul(m[13], p3));
	ret[2] = FTD(fixmul(m[2], p0) + fixmul(m[6], p1) +
		fixmul(m[10], p2) + fixmul(m[14], p3));
	ret[3] = FTD(fixmul(m[3], p0) + fixmul(m[7], p1) +
		fixmul(m[11], p2) + fixmul(m[15], p3));
#else
	GLfloat p0,p1,p2,p3;
	
	p0 = p[0];
	p1 = p[1];
	p2 = p[2];
	p3 = p[3];
	
	ret[0] = m[0] * p0 + m[4] * p1 + m[8] * p2 + m[12] * p3;
	ret[1] = m[1] * p0 + m[5] * p1 + m[9] * p2 + m[13] * p3;
	ret[2] = m[2] * p0 + m[6] * p1 + m[10] * p2 + m[14] * p3;
	ret[3] = m[3] * p0 + m[7] * p1 + m[11] * p2 + m[15] * p3;
#endif
}

/*
 * v = p1 - p0
 */
static void VectorMinusVector(const GLfloat *p1, const GLfloat *p0, GLfloat *v) {
	int i;

	for (i=0;i<4;i++) {
		v[i] = p1[i] - p0[i];
	}
}

/*
 * returns the dot product of vectors v1 and v2.
 */
static GLfloat VectorDotVector(const GLfloat *v1, const GLfloat *v2) {
	GLfloat answer;
	
	answer = v1[0]*v2[0];
	answer += v1[1]*v2[1];
	answer += v1[2]*v2[2];
	answer += v1[3]*v2[3];

	return answer;
}

static void VectorNormalize(GLfloat *v) {
	const float l_sqr = v[0]*v[0] +
		v[1]*v[1] +
		v[2]*v[2] +
		v[3]*v[3];
	const float l_inv = 1/Sqrt(l_sqr);

	v[0] = v[0]*l_inv;
	v[1] = v[1]*l_inv;
	v[2] = v[2]*l_inv;
	v[3] = v[3]*l_inv;
}

/*
 * cross = c0 x c1 (cross product).
 */
static void VectorCrossVector(const GLfloat *v1, const GLfloat *v2, GLfloat *cross) {
	cross[0] = v1[1]*v2[2] - v2[1]*v1[2];
	cross[1] = v1[2]*v2[0] - v2[2]*v1[0];
	cross[2] = v1[0]*v2[1] - v2[0]*v1[1];
	cross[3] = 1;
}

/*
 * Initialize a 4x4 matrix to the identity matrix (all zeroes except ones in
 * the diagonal).
 *
 * m - input 4x4 matrix 
 */
static void InitializeMatrix(fixed *m) {
	int i;

	for (i=0;i<16;i++) 
		m[i] = 0.0;

	m[0] = 1;
	m[5] = 1;
	m[10] = 1;
	m[15] = 1;
}

static void MatrixToFixed(const GLfloat* d, fixed* f) {
	int i = 0;
	for (; i < 16; i++)
		f[i] = DTF(d[i]);
}

/*
 * From Matt Wronkiewicz
 */
/*
void glMultMatrixf(const GLfloat *input) {
	fixed *m = cur_matrix;
	fixed c[16];
	
	if (m == NULL)
		return;

#ifdef FIXED_MATH
	MatrixToFixed(input, c);
#endif
	MatrixMultFull(c, m);
}
*/
void glMultMatrixf(const GLfloat *input) {
        GLfloat result[16];
        GLfloat *c;
        GLfloat *m;
        int i,j;

        c = input;
        m = cur_matrix;

        /** 4x4 matrix multiplication */
        for (j=0;j<4;j++) {
                result[0+j] = c[0+j]*m[0] + c[4+j]*m[1]
                                        + c[8+j]*m[2] + c[12+j]*m[3];
                result[4+j] = c[0+j]*m[4] + c[4+j]*m[5]
                                        + c[8+j]*m[6] + c[12+j]*m[7];
                result[8+j] = c[0+j]*m[8] + c[4+j]*m[9]
                                        + c[8+j]*m[10] + c[12+j]*m[11];
                result[12+j] = c[0+j]*m[12] + c[4+j]*m[13]
                                        + c[8+j]*m[14] + c[12+j]*m[15];
        }

        for (i=0;i<16;i++) {
                cur_matrix[i] = result[i];
        }
}


/*
static void MatrixMultFull(const fixed* c, fixed* m) {
	fixed temp1;
	fixed temp2;
	fixed temp3;
	
#ifdef FIXED_MATH
	temp1 = fixmul(c[0], m[0]) + fixmul(c[4], m[1]) +
		fixmul(c[8], m[2]) + fixmul(c[12], m[3]);
	temp2 = fixmul(c[1], m[0]) + fixmul(c[5], m[1]) +
		fixmul(c[9], m[2]) + fixmul(c[13], m[3]);
	temp3 = fixmul(c[2], m[0]) + fixmul(c[6], m[1]) +
		fixmul(c[10], m[2]) + fixmul(c[14], m[3]);
	m[3] = fixmul(c[3], m[0]) + fixmul(c[7], m[1]) +
		fixmul(c[11], m[2]) + fixmul(c[15], m[3]);
	m[2] = temp3;
	m[1] = temp2;
	m[0] = temp1;
	temp1 = fixmul(c[0], m[4]) + fixmul(c[4], m[5]) +
		fixmul(c[8], m[6]) + fixmul(c[12], m[7]);
	temp2 = fixmul(c[1], m[4]) + fixmul(c[5], m[5]) +
		fixmul(c[9], m[6]) + fixmul(c[13], m[7]);
	temp3 = fixmul(c[2], m[4]) + fixmul(c[6], m[5]) +
		fixmul(c[10], m[6]) + fixmul(c[14], m[7]);
	m[7] = fixmul(c[3], m[4]) + fixmul(c[7], m[5]) +
		fixmul(c[11], m[6]) + fixmul(c[15], m[7]);
	m[6] = temp3;
	m[5] = temp2;
	m[4] = temp1;
	temp1 = fixmul(c[0], m[8]) + fixmul(c[4], m[9]) +
		fixmul(c[8], m[10]) + fixmul(c[12], m[11]);
	temp2 = fixmul(c[1], m[8]) + fixmul(c[5], m[9]) +
		fixmul(c[9], m[10]) + fixmul(c[13], m[11]);
	temp3 = fixmul(c[2], m[8]) + fixmul(c[6], m[9]) +
		fixmul(c[10], m[10]) + fixmul(c[14], m[11]);
	m[11] = fixmul(c[3], m[8]) + fixmul(c[7], m[9]) +
		fixmul(c[11], m[10]) + fixmul(c[15], m[11]);
	m[10] = temp3;
	m[9] = temp2;
	m[8] = temp1;
	temp1 = fixmul(c[0], m[12]) + fixmul(c[4], m[13]) +
		fixmul(c[8], m[14]) + fixmul(c[12], m[15]);
	temp2 = fixmul(c[1], m[12]) + fixmul(c[5], m[13]) +
		fixmul(c[9], m[14]) + fixmul(c[13], m[15]);
	temp3 = fixmul(c[2], m[12]) + fixmul(c[6], m[13]) +
		fixmul(c[10], m[14]) + fixmul(c[14], m[15]);
	m[15] = fixmul(c[3], m[12]) + fixmul(c[7], m[13]) +
		fixmul(c[11], m[14]) + fixmul(c[15], m[15]);
	m[14] = temp3;
	m[13] = temp2;
	m[12] = temp1;
#else
	temp1 = c[0]* m[0] + c[4]* m[1] + c[8]* m[2] + c[12]* m[3];
	temp2 = c[1]* m[0] + c[5]* m[1] + c[9]* m[2] + c[13]* m[3];
	temp3 = c[2]* m[0] + c[6]* m[1] + c[10]* m[2] + c[14]* m[3];
	m[3] = c[3]* m[0] + c[7]* m[1] + c[11]* m[2] + c[15]* m[3];
	m[2] = temp3;
	m[1] = temp2;
	m[0] = temp1;
	temp1 = c[0]* m[4] + c[4]* m[5] + c[8]* m[6] + c[12]* m[7];
	temp2 = c[1]* m[4] + c[5]* m[5] + c[9]* m[6] + c[13]* m[7];
	temp3 = c[2]* m[4] + c[6]* m[5] + c[10]* m[6] + c[14]* m[7];
	m[7] = c[3]* m[4] + c[7]* m[5] + c[11]* m[6] + c[15]* m[7];
	m[6] = temp3;
	m[5] = temp2;
	m[4] = temp1;
	temp1 = c[0]* m[8] + c[4]* m[9] + c[8]* m[10] + c[12]* m[11];
	temp2 = c[1]* m[8] + c[5]* m[9] + c[9]* m[10] + c[13]* m[11];
	temp3 = c[2]* m[8] + c[6]* m[9] + c[10]* m[10] + c[14]* m[11];
	m[11] = c[3]* m[8] + c[7]* m[9] + c[11]* m[10] + c[15]* m[11];
	m[10] = temp3;
	m[9] = temp2;
	m[8] = temp1;
	temp1 = c[0]* m[12] + c[4]* m[13] + c[8]* m[14] + c[12]* m[15];
	temp2 = c[1]* m[12] + c[5]* m[13] + c[9]* m[14] + c[13]* m[15];
	temp3 = c[2]* m[12] + c[6]* m[13] + c[10]* m[14] + c[14]* m[15];
	m[15] = c[3]* m[12] + c[7]* m[13] + c[11]* m[14] + c[15]* m[15];
	m[14] = temp3;
	m[13] = temp2;
	m[12] = temp1;
#endif
}
*/

/* Multiplies a 4x4 matrix by the current matrix but assumes that
 * the matrix passed in is 3-D. */
/*
static void MatrixMultNoTranslate(const fixed* c, fixed* m) {
	fixed temp1;
	fixed temp2;

	temp1 = fixmul(c[0], m[0]) + fixmul(c[4], m[1]) + fixmul(c[8], m[2]);
	temp2 = fixmul(c[1], m[0]) + fixmul(c[5], m[1]) + fixmul(c[9], m[2]);
	m[2] = fixmul(c[2], m[0]) + fixmul(c[6], m[1]) + fixmul(c[10], m[2]);
	m[1] = temp2;
	m[0] = temp1;
	temp1 = fixmul(c[0], m[4]) + fixmul(c[4], m[5]) + fixmul(c[8], m[6]);
	temp2 = fixmul(c[1], m[4]) + fixmul(c[5], m[5]) + fixmul(c[9], m[6]);
	m[6] = fixmul(c[2], m[4]) + fixmul(c[6], m[5]) + fixmul(c[10], m[6]);
	m[5] = temp2;
	m[4] = temp1;
	temp1 = fixmul(c[0], m[8]) + fixmul(c[4], m[9]) + fixmul(c[8], m[10]);
	temp2 = fixmul(c[1], m[8]) + fixmul(c[5], m[9]) + fixmul(c[9], m[10]);
	m[10] = fixmul(c[2], m[8]) + fixmul(c[6], m[9]) + fixmul(c[10], m[10]);
	m[9] = temp2;
	m[8] = temp1;
	temp1 = fixmul(c[0], m[12]) + fixmul(c[4], m[13]) + fixmul(c[8], m[14]);
	temp2 = fixmul(c[1], m[12]) + fixmul(c[5], m[13]) + fixmul(c[9], m[14]);
	m[14] = fixmul(c[2], m[12]) + fixmul(c[6], m[13]) + fixmul(c[10], m[14]);
	m[13] = temp2;
	m[12] = temp1;
}
*/

/*
 * Loads the input matrix m into the current matrix, overwriting any values
 * stored in the current matrix.
 *
 * m - input 4x4 matrix
 */
/*
void glLoadMatrixf(const GLfloat *in) {
	fixed *m = cur_matrix;
	int i;

	if (m == NULL)
		return;

	for (i=0;i<16;i++) {
#ifdef FIXED_MATH
		m[i] = DTF(in[i]);
#else
		m[i] = in[i];
#endif
	}
}
*/
void glLoadMatrixf(const GLfloat *m) {
        int i;

        for (i=0;i<16;i++) {
                cur_matrix[i] = m[i];
        }
}


/*
 * The gl call to load in the identity matrix as the current geometry transform
 * matrix.  Same as InitializeMatrix(...) but uses the gl function name.
 */
void glLoadIdentity() {
	fixed *m = cur_matrix;

	if (m == NULL)
		return;

	InitializeMatrix(cur_matrix);
}

/*
 * Doesn't do anything right now since there are no colors.
 */
void glClearColor(GLclampf red, GLclampf green, GLclampf blue, GLclampf alpha) {
}

/*
 * Inititalize z-buffer.
 */
void ZBufferInit() {
	int j;

	/** create z-buffer */
	/** Only use scan-line zbuffer to save heap space */
	zbuffer = (GLubyte *) MemChunkNew(MemHeapID(0, 0), 
				MAX_SCREEN_WIDTH, // * MAX_SCREEN_HEIGHT,
				memNewChunkFlagNonMovable);
				/** remember to clear this */
	
	/** initialize to deepest values */
	for (j=0;j<160;j++) {
		zbuffer[j] = 0xff; // set to largest (deepest) value
	}
}

/*
 * Clears the gl viewport as specified through the glViewport(...) call.  
 * Basically draws a "white" rectangle over the whole region to clear it.
 *
 * mask - buffer to clear 
 */
void glClear(GLbitfield mask) {
	Err err;
	GLint i,j;

	if (mask != GL_COLOR)
		return; /** anything else not supported now */

	WinSetDrawWindow(two);
	fgClearWindow(screen_startx, screen_starty, screen_width,
					screen_height);
	WinSetDrawWindow(one);

}

static void GetCurColor(RGBColorType *color1) {  
	color1->r = cur_color[0];
	color1->g = cur_color[1];
	color1->b = cur_color[2];
}

/*
 * Set the current color to be the RGB value (r,g,b) with an alpha value of 1.
 *
 * r - red [0,1]
 * g - green [0,1]
 * b - blue [0,1]
 */
void glColor3f(GLfloat r, GLfloat g, GLfloat b) {

	cur_color[0] = r;
	cur_color[1] = g;
	cur_color[2] = b;
	cur_color[3] = 1.0;

	fgSetColor(r,g,b);
}

/*
 * Set the current color to be the RGB value (r,g,b) with an alpha value of a.
 *
 * r - red [0,1]
 * g - green [0,1]
 * b - blue [0,1]
 * a - alpha transparency [0,1]
 */
void glColor4f(GLfloat r, GLfloat g, GLfloat b, GLfloat a) {

	cur_color[0] = r;
	cur_color[1] = g;
	cur_color[2] = b;
	cur_color[3] = a;

	fgSetColor(r,g,b);
}

/*
 * Sets up the perspective projection matrix and screen coord translation 
 * matrix to be an orthographic projection (no perspective).
 *
 * left - world x coordinate of left edge of projection plane
 * right - world x coordinate of right edge of projection plane
 * bottom - world y coordinate of bottom edge of projection plane
 * top - world y coordinate of top edge of projection plane
 * near - world z coordinate of distance from camera to projection plane
 * distant - back clipping plane
 */
void glOrtho(GLdouble left, GLdouble right, GLdouble bottom, GLdouble top,
		GLdouble near, GLdouble distant) {
	//fixed *m = cur_matrix;
	//fixed tmp_mat[16];
	float d;

	const GLfloat width = right - left;
	const GLfloat height = top - bottom;
	
	//if (m == NULL)
	//	return;

	//InitializeMatrix(tmp_mat);
	//tmp_mat[0] = DTF(2.0/width);
	//tmp_mat[5] = DTF(2.0/height);
	//tmp_mat[10] = DTF(-2.0/(distant - near));
	//tmp_mat[12] = -DTF(((float)right + left)/width);
	//tmp_mat[13] = -DTF(((float)top + bottom)/height);
	//tmp_mat[14] = -DTF(((float)distant + near)/(distant - near));
	//MatrixMultFull(tmp_mat, m);

	d = 32000;
	InitializeMatrix(per_matrix);
        per_matrix[10] = 0;
        per_matrix[11] = (1.0/d);

	InitializeMatrix(scr_matrix);
	scr_matrix[0] = DTF((float)screen_width / width);
	scr_matrix[5] = -DTF((float)screen_height / height);
	scr_matrix[10] = 0;
	scr_matrix[12] = ITF(screen_startx) + ITF(screen_width)/2;
	scr_matrix[13] = ITF(screen_starty) + ITF(screen_height)/2;
}

/*
 * Sets up the perspective projection matrix and screen coord translation 
 * matrix to be a 2D orthographic projection (no perspective).  Distance from
 * camera to projection plane tends to infinity.
 *
 * left - world x coordinate of left edge of projection plane
 * right - world x coordinate of right edge of projection plane
 * bottom - world y coordinate of bottom edge of projection plane
 * top - world y coordinate of top edge of projection plane
 */
void gluOrtho2D(GLdouble left, GLdouble right, GLdouble bottom, GLdouble top) {
	glOrtho(left, right, bottom, top, -1, 1);
}

/*
 * Sets up the viewport.  Same stuff as WinSetGLArea below.
 */
void glViewport(GLint x, GLint y, GLsizei width, GLsizei height) {
	// fill in here.
}

/* 
 * Sets up the perspective projection matrix and screen coordinate transform
 * matrix to correctly render perspective projections given a field of view
 * fovy and an aspect ratio aspect (x/y).  Note:  GLU function.
 *
 * fovy - field of view in degrees [0,180]
 * aspect - aspect ratio as width/height (standard television is 4/3)
 * near - distance from camera to projection plane (world coords)
 * distant - back clipping plane world z value
 */
void gluPerspective(GLdouble fovy, GLdouble aspect, GLdouble near, 
			GLdouble distant) {
	fixed *m = cur_matrix;
	fixed tmp_mat[16];
	const float half_angle = (float)(fovy > 0?fovy:45.0)*PI/360.0; // convert to radians
	const float f = Cos(half_angle)/Sin(half_angle);
	const float height = 2*near/f;
	const float width  = height*(aspect > 0?aspect:1.0);
	
	if (m == NULL)
		return;

	//InitializeMatrix(tmp_mat);
	//tmp_mat[0] = DTF(f/(aspect > 0?aspect:1.0));
	//tmp_mat[5] = DTF(f);
	//tmp_mat[10] = DTF(((float)distant + near)/(near - distant));
	//tmp_mat[11] = 1; /*ITF(1);  Spec says this should be -1. */
	//tmp_mat[14] = fixmul(ITF(2), tmp_mat[10]);
	//tmp_mat[15] = 0; /*Currently crashes, but this is what the spec says. */
	//MatrixMultFull(tmp_mat, m);

	InitializeMatrix(per_matrix);
        per_matrix[10] = 0;
        per_matrix[11] = (1.0/near);

	InitializeMatrix(scr_matrix);
	scr_matrix[0] = DTF((float)screen_width / width);
	scr_matrix[5] = -DTF((float)screen_height / height);
	scr_matrix[10] = 0;
	scr_matrix[12] = ITF(screen_startx) + ITF(screen_width)/2;
	scr_matrix[13] = ITF(screen_starty) + ITF(screen_height)/2;
}

/*
 * Begin a graphics primitive.
 */
void glBegin(GLenum mode) {
	num_vertices = 0;
	cur_mode = mode;
}

/** Special function for BackFacing */
static void MatrixMultVector2D(const fixed *m, const GLfloat *p,
		GLfloat *ret) {
	const fixed p0 = DTF(p[0]);
	const fixed p1 = DTF(p[1]);
	const fixed p2 = DTF(p[2]);
	const fixed p3 = DTF(p[3]);

	ret[0] = FTD(fixmul(m[0], p0) + fixmul(m[4], p1) +
		fixmul(m[8], p2) + fixmul(m[12], p3));
	ret[1] = FTD(fixmul(m[1], p0) + fixmul(m[5], p1) +
		fixmul(m[9], p2) + fixmul(m[13], p3));
	ret[3] = FTD(fixmul(m[3], p0) + fixmul(m[7], p1) +
		fixmul(m[11], p2) + fixmul(m[15], p3));
}

/*
 * Check if it's facing away from the camera in order to determine whether or
 * not to draw it if culling is turned on.
 */
static int BackFacing(int l, int m, int n) {
	const GLfloat* q0 = scr_vertices[l];
	const GLfloat* q1 = scr_vertices[m];
	const GLfloat* q2 = scr_vertices[n];

	/** Then use cross product to find direction it's facing */
	/*VectorMinusVector(q1, q0, c0);
	VectorMinusVector(q2, q0, c1);*/
	/** Reduce size to keep from overflowing (/8) */
	const fixed c00 = (DTF(q1[0]) - DTF(q0[0]))/8;
	const fixed c01 = (DTF(q1[1]) - DTF(q0[1]))/8;
	const fixed c10 = (DTF(q2[0]) - DTF(q0[0]))/8;
	const fixed c11 = (DTF(q2[1]) - DTF(q0[1]))/8;
	/** Partial cross product replaces VectorCrossVector */ 	
	
	/** Return (cross > 0) */
	return fixmul(c00, c11) > fixmul(c10, c01);
}               


static void DoLightingCalc(GLfloat pos[4], GLfloat normal[4], 
				RGBColorType *color1) {
	GLfloat blaa[4];
	GLfloat cosangle;
	int i;

	if (!lighting) {
		color1->r = cur_color[0];
		color1->g = cur_color[1];
		color1->b = cur_color[2];

		return;
	}
	
	for (i=0;i<8;i++) { /** Go through all eight lights */
		if (lights[i].enabled == 0)
			break;

		VectorMinusVector(lights[i].position, pos, blaa);
		VectorNormalize(normal);
		VectorNormalize(blaa);
		cosangle = VectorDotVector(normal, blaa);
		if (cosangle < 0) {
			color1->r = cur_color[0] * lights[i].ambient[0];
			color1->g = cur_color[1] * lights[i].ambient[1];
			color1->b = cur_color[2] * lights[i].ambient[2];
		} else {
			color1->r = cur_color[0] *(lights[i].ambient[0] 
					+ (lights[i].diffuse[0]
					* cosangle));
			color1->g = cur_color[1] *(lights[i].ambient[1]
					+ (lights[i].diffuse[1]
					* cosangle));
			color1->b = cur_color[2]  *(lights[i].ambient[2]
					+ (lights[i].diffuse[2]
					* cosangle));
		}
	}
}

/*
 * Draw a line using the z-buffer and shading.
 */
static void DrawScanLine(GLfloat *start, GLfloat *end, GLfloat *startnormal,
			GLfloat *endnormal, int mode) {
	RGBColorType color1, color2, color;
	int ty, tx, tz;
	int i,j;
	GLfloat temp[4], normal[4];

#ifdef ZBUFFER
	/* 
	 * Do bidness here to draw point by point; test and set z-buffer 
	 * along the way.  Put shading calcs in here, too, by interpolating
	 * the normal/intensities across the line.  Add in a beginning value
	 * and ending value to the two passed points to interpolate.  If
	 * doing Phong shading, interpolate normal (expensive!) or for now
	 * just do Gouraud shading by interpolating color values given at
	 * each end of the lines.
	 */

	temp[3] = 1;
	temp[1] = start[2];
	
	if (mode == GOURAUD) {
		DoLightingCalc(start, startnormal, &color1);
		DoLightingCalc(end, endnormal, &color2);
	}

	for (j=end[0];j<=start[0];j++) {
		if (mode == PHONG) {
			temp[0] = j;
			temp[2] = start[3]; // interpolated z

			/** Change normal for Phong shading */
			for (i=0;i<4;i++) {
				//normal[i] = normal[i];
			}

			/** Do lighting calc */
			DoLightingCalc(temp, normal, &color);
		} else {
		    /** interpolate between color1 and color2 */
		    color.r = ((j-start[0])/(end[0]-start[0]))*color2.r 
			+ (1-(j-start[0])/(end[0]-start[0]))*color1.r;
		    color.g = ((j-start[0])/(end[0]-start[0]))*color2.g 
			+ (1-(j-start[0])/(end[0]-start[0]))*color1.g;
		    color.b = ((j-start[0])/(end[0]-start[0]))*color2.b 
			+ (1-(j-start[0])/(end[0]-start[0]))*color1.b;
		}

		fgSetColor(color.r, color.g, color.b);
		// if (z bidness)
		//	do stuff

		fgDrawPixel(j,start[1]);
	}
#else
	/** Just draw the 2D scan line */
	fgDrawLine(start[0], start[1], end[0], end[1]);
#endif
}

static void SetColor() {
	RGBColorType color;

	GetCurColor(&color);
	fgSetColor(color.r, color.g, color.b);
}

static void TransformToScreen(const GLfloat* in, GLfloat* ret) {
	const fixed* m = scr_matrix;
	const fixed p0 = DTF(in[0]);
	const fixed p1 = DTF(in[1]);
	const fixed p2 = DTF(in[2]);
	const fixed p3 = DTF(in[3]);

	ret[0] = FTD(fixmul(m[0], p0) + fixmul(m[4], p1) +
		fixmul(m[8], p2) + fixmul(m[12], p3));
	ret[1] = FTD(fixmul(m[1], p0) + fixmul(m[5], p1) +
		fixmul(m[9], p2) + fixmul(m[13], p3));
	ret[2] = FTD(fixmul(m[2], p0) + fixmul(m[6], p1) +
		fixmul(m[10], p2) + fixmul(m[14], p3));
}

/*
 * End a graphics primitive and draw to the buffer.
 */
#define W 1
#define EPSILON 0.02
void glEnd(void) {
	static int i, sx1, sx2, sy1, sy2, j, oldtopy, oldbottomy, oldi, oldj,
		o, n, y, tempx, tempy, q;
	static float x1, y1, x2, y2, cosangle;
	static GLfloat  p1[4], in1[4], out1[4], normal[4], start[4],
			end[4];
	static RGBColorType color1, color2;
	/** In no case is the fourth element used. */
	/*static GLfloat scr_vertices[MAX_VERTICES][3];*/
	static Line I, II;

	if (num_vertices < 2)
		return;

	WinSetDrawWindow(two);
	WinSetActiveWindow(one);

	/** Translate vertices to screen coords */
	for(i=0; i < num_vertices; i++) {
		/** Geometric transform */
		//MatrixMultVector(modv_matrix[modv_level], vertices[i], p1);
		MatrixMultVector(cur_matrix, vertices[i], p1);

		/** Perspective transform */
		//MatrixMultVector(proj_matrix[proj_level], p1, in1);
		MatrixMultVector(per_matrix, p1, in1);

		/** Go from 4D back to 3D */
		for(j=0;j<4;j++) {
			in1[j] = in1[j]/in1[3];
		}

		/** Translate into screen coords */
		//TransformToScreen(in1, scr_vertices[i]);

                MatrixMultVector(scr_matrix, in1, out1);

                /** Add the point to the final list */
                for(j=0;j<4;j++) {
                        scr_vertices[i][j] = out1[j];
                }

	}
	//MatrixMultVector(modv_matrix[modv_level], cur_normal, normal);
	MatrixMultVector(cur_matrix, cur_normal, normal);
	
	switch(cur_mode) {
		case GL_TRIANGLES:
			if (num_vertices < 3)
				break;

			SetColor();

			j = 0;
			while ((j+2) < num_vertices) {
				if (culling && !BackFacing(j,j+1,j+2)) {
					for (o=j+1;o<(j+3);o++) {
						fgDrawLine(
							scr_vertices[o-1][0],
							scr_vertices[o-1][1],
							scr_vertices[o][0],
							scr_vertices[o][1]);
					}
					fgDrawLine(
						scr_vertices[o-1][0],
						scr_vertices[o-1][1],
						scr_vertices[j][0],
						scr_vertices[j][1]);
				}
				j+=3;
			}
			break;

		case GL_TRIANGLE_STRIP:
			if (num_vertices < 3)
				break;

			SetColor();

			/** Draw first line between verts 0 and 1 */
			fgDrawLine( 
				scr_vertices[0][0],
				scr_vertices[0][1],
				scr_vertices[1][0],
				scr_vertices[1][1]);

			/** Need to add culling support */

			o = 2;
			/** For each new vertex, draw two new triangle legs */
			while (o < num_vertices) {
				
				fgDrawLine( 
					scr_vertices[o][0],
					scr_vertices[o][1],
					scr_vertices[o-1][0],
					scr_vertices[o-1][1]);
				fgDrawLine(
					scr_vertices[o][0],
					scr_vertices[o][1],
					scr_vertices[o-2][0],
					scr_vertices[o-2][1]);

				o++;
			}
			break;

		case GL_TRIANGLE_FAN:
			if (num_vertices < 3)
				break;

			SetColor();

			for(i=1; i < num_vertices; i++) {
				if (culling && !BackFacing(0,i-1,i)) {
                                	fgDrawLine(scr_vertices[0][0],
						scr_vertices[0][1],
						scr_vertices[0][2],
						scr_vertices[i][1]);
					fgDrawLine(scr_vertices[i-1][0],
						scr_vertices[i-1][1],
						scr_vertices[i][0],
						scr_vertices[i][1]);
				}
			}
			break;

		case GL_QUADS:
			if (num_vertices < 4)
				break;

			SetColor();

			j = 0;
			while ((j+3) < num_vertices) {

				if (culling && !BackFacing(j,j+1,j+2)) {

				// draw bidness with j, j+1, j+2, j+3 vertices
					for (o=j+1;o<(j+4);o++) {
						fgDrawLine(
							scr_vertices[o-1][0],
							scr_vertices[o-1][1],
							scr_vertices[o][0],
							scr_vertices[o][1]);
					}
					fgDrawLine(
						scr_vertices[o-1][0],
						scr_vertices[o-1][1],
						scr_vertices[j][0],
						scr_vertices[j][1]);
				}
				j+=4;
			}
			break;

		case GL_QUAD_STRIP:
			SetColor();

			/** Do this */
			break;

		case GL_LINE_LOOP:
			SetColor();

			if (num_vertices < 1)
				break;

			for(i=1; i < num_vertices; i++) {
                                fgDrawLine(scr_vertices[i-1][0],
					scr_vertices[i-1][1],
					scr_vertices[i][0],
					scr_vertices[i][1]);
			}
			/** Close the loop */
			fgDrawLine(scr_vertices[i-1][0],
				scr_vertices[i-1][1],
				scr_vertices[0][0],
				scr_vertices[0][1]);
			break;

		case GL_LINES:
			SetColor();

			if (num_vertices < 1)
				break;

			for(i=1; i < num_vertices; i+=2) {
                                fgDrawLine(scr_vertices[i-1][0],
					scr_vertices[i-1][1],
					scr_vertices[i][0],
					scr_vertices[i][1]);
			}
			break;

		case GL_POINTS:
			SetColor();

			for(i=0; i < num_vertices; i++) {
				fgDrawLine(scr_vertices[i][0],
					scr_vertices[i][1],
					scr_vertices[i][0],
					scr_vertices[i][1]);
			}
			break;

		case GL_LINE_STRIP:
			SetColor();

			for(i=1; i < num_vertices; i++) {
                                fgDrawLine(scr_vertices[i-1][0],
					scr_vertices[i-1][1],
					scr_vertices[i][0],
					scr_vertices[i][1]);
			}
			break;

		/** Right now only GL_POLYGON is floodfilled and lit */
		case GL_POLYGON:
			if (num_vertices < 3)
				break;

			if (culling && BackFacing(0,1,2)) {
				break;
			}
			
			/*
			 * For now, floodfill is only applied to GL_POLYGON
			 * type for testing.
			 */

			if (!wireframe) {
			/** lighting calculation to get color */
			DoLightingCalc(vertices[0], normal, &color1);
			fgSetColor(color1.r, color1.g, color1.b);

			/** find "highest" and "lowest" points in y dir */
			oldtopy = -1000;
			oldbottomy = 1000;
			for (i=0;i<num_vertices;i++) {
				if (scr_vertices[i][1] > oldtopy) {
					oldtopy = scr_vertices[i][1];
					oldi = i;
				}
				if (scr_vertices[i][1] < oldbottomy) {
					oldbottomy = scr_vertices[i][1];
				}
			}

			/** Flood fill polygon */
			n = num_vertices;

			/** Set first set of boundary lines */
			I.start = (oldi) % n;
			I.end = (oldi+1) % n;
			I.x1 = scr_vertices[I.start][0];
			I.y1 = scr_vertices[I.start][1];
			I.z1 = scr_vertices[I.start][2];
			I.m = (GLfloat)(scr_vertices[I.end][1]-I.y1) 
			    / (GLfloat)(scr_vertices[I.end][0]-I.x1);
			I.mz = (GLfloat)(scr_vertices[I.end][1]-I.y1) 
			    / (GLfloat)(scr_vertices[I.end][2]-I.z1);

			II.start = (oldi) % n;
			II.end = (oldi+n-1) % n;
			II.x1 = scr_vertices[II.start][0];
			II.y1 = scr_vertices[II.start][1];
			II.z1 = scr_vertices[II.start][2];
			II.m = (GLfloat)(scr_vertices[II.end][1]-II.y1)
			     / (GLfloat)(scr_vertices[II.end][0]-II.x1);
			II.mz = (GLfloat)(scr_vertices[II.end][1]-II.y1)
			     / (GLfloat)(scr_vertices[II.end][2]-II.z1);
			
			/** Loop over all y values between "top" and "bottom" */
			for (y = oldtopy-1; y > oldbottomy; y--) {
				/** Draw one scanline */
				if (I.m > -EPSILON && I.m < EPSILON) {
					start[0] = scr_vertices[I.end][0];
					start[2] = scr_vertices[I.end][2];
				} else {
					start[0] = I.x1 
					       + (int)((GLfloat)(y-I.y1)/I.m);
					start[2] = I.z1 
					       + (int)((GLfloat)(y-I.y1)/I.mz);
				}
				if (II.m > -EPSILON && II.m < EPSILON) {
					end[0] = scr_vertices[II.end][0];
					end[2] = scr_vertices[II.end][2];
				} else {
					end[0] = II.x1 
					     + (int)((GLfloat)(y-II.y1)/II.m);
					end[2] = II.z1 
					     + (int)((GLfloat)(y-II.y1)/II.mz);
				}
				start[1] = y;
				start[3] = 1;
				end[1] = y;
				end[3] = 1;
				DrawScanLine(start, end, normal, normal, 
						GOURAUD);

				/** Check if the boundary lines should change */
				tempy = scr_vertices[I.end][1];
				if (y>=tempy-W && y<=tempy+W) {
					/** change I */
					oldi++;
					I.start = I.end;
					I.end = (I.end+1) % n;
					I.x1 = scr_vertices[I.start][0];
					I.y1 = scr_vertices[I.start][1];
					I.m = (GLfloat)
						(scr_vertices[I.end][1]-I.y1)/
						(GLfloat)
						(scr_vertices[I.end][0]-I.x1);
				} 

				tempy = scr_vertices[II.end][1];
				if (y>=tempy-W && y<=tempy+W) {
					/** change II */
					II.start = II.end;
					II.end = (II.end+n-1) % n;
					II.x1 = scr_vertices[II.start][0];
					II.y1 = scr_vertices[II.start][1];
					II.m = (GLfloat)
						(scr_vertices[II.end][1]-II.y1)/
						(GLfloat)
						(scr_vertices[II.end][0]-II.x1);
				}
			}
			}
	
			/** Draw black edge lines */
			if (wireframe) {
				fgSetColor(0,0,0);
				for (i=1;i<num_vertices;i++) {
					fgDrawLine(scr_vertices[i-1][0],
						scr_vertices[i-1][1],
						scr_vertices[i][0],
						scr_vertices[i][1]);
				}
				fgDrawLine(scr_vertices[i-1][0],
					scr_vertices[i-1][1],
					scr_vertices[0][0],
					scr_vertices[0][1]);
			}
			break;

		default:
			break;
	}
	num_vertices = 0;

	//fgSetColor(0,0,0);
	//fgDrawLine(0,0,160,160);

	WinSetDrawWindow(one);
	WinSetActiveWindow(one);
}

/*
 * Copies the back buffer contents into the screen bitmap area to display the
 * drawn primitives.  Only copies the area as set in WinSetGLArea(...).
 */
void glutSwapBuffers() {
	glFlush();
}

/*
 * For now does the same as glutSwapBuffers() above.
 */
void glFlush(void) {
	static RectangleType rect;
	RGBColorType color1, color2;

#ifdef PRINT_FPS
	ULong t;
	static ULong old_t;
	static int count_time=0;
	int fps;
	static int framecount;
	char text[30];
#endif

	rect.topLeft.x = screen_startx;
	rect.topLeft.y = screen_starty;
	rect.extent.x = screen_width;
	rect.extent.y = screen_height;

#ifdef PRINT_FPS
	/** Don't count time spent swapping buffers now. */
	t = TimGetTicks();
#endif
	if (two) {
		WinCopyRectangle(two, one, &rect, 
				screen_startx, screen_starty, scrCopy);
	}


	fgSetColor(100,100,100);

#ifdef PRINT_FPS
	framecount++;
	if (t - old_t > 0 && count_time) {
		//fps = framecount;
		fps = (t - old_t)*1000/SysTicksPerSecond();
		StrIToA(text, fps);
		StrCat(text, " mspf    ");
		WinDrawChars(text, StrLen(text), 100, 0);
		framecount=0;
		old_t = t;
	} else if (!count_time) {
		old_t = t;
		count_time = 1;
	}
#endif

	fgSetColor(0,0,0);
}

/*
 * 
 */
void glVertex2f(GLfloat x, GLfloat y) {
	glVertex3f(x, y, 0.0);
}

/*
 * 
 */
void glVertex3f(GLfloat x, GLfloat y, GLfloat z) {
	int i;

	if (num_vertices < MAX_VERTICES) {
		vertices[num_vertices][0] = x;
		vertices[num_vertices][1] = y;
		vertices[num_vertices][2] = z;
		vertices[num_vertices][3] = 1.0;

		for (i=0;i<4;i++) {
			vertices_color[num_vertices][i] = cur_color[i];
			vertices_normal[num_vertices][i] = cur_normal[i];
		}

		num_vertices++;
	}
}

/*
 * 
 */
void glShadeModel(GLenum mode) {
	cur_shade_model = GL_FLAT;
}

/*
 * 
 */
void glRotatef(GLfloat angle, GLfloat x, GLfloat y, GLfloat z) {
/*
	fixed *m = cur_matrix;
	fixed tmp[16];
	const float theta = angle*PI/180.0;
	const fixed s = DTF(Sin(theta));
	const fixed c = DTF(Cos(theta));
	const fixed t = ITF(1) - c;
	const fixed fx = DTF(x);
	const fixed fy = DTF(y);
	const fixed fz = DTF(z);
	const fixed tx = fixmul(t, fx);
	const fixed ty = fixmul(t, fy);
	const fixed tz = fixmul(t, fz);
	const fixed sx = fixmul(s, fx);
	const fixed sy = fixmul(s, fy);
	const fixed sz = fixmul(s, fz);
	
	if (m == NULL)
		return;
	
	tmp[0] = fixmul(tx, fx) + c;
	tmp[1] = fixmul(tx, fy) + sz;
	tmp[2] = fixmul(tx, fz) - sy;
	tmp[4] = fixmul(ty, fx) - sz;
	tmp[5] = fixmul(ty, fy) + c;
	tmp[6] = fixmul(ty, fz) + sx;
	tmp[8] = fixmul(tz, fx) + sy;
	tmp[9] = fixmul(tz, fy) - sx;
	tmp[10] = fixmul(tz, fz) + c;

	MatrixMultFull(tmp, m);
*/
        GLfloat t[16];
        double theta;

        theta = (angle/180.0) * PI;

        InitializeMatrix(t);

        if (z == 1) {
                t[0] = Cos(theta);
                t[4] = -Sin(theta);
                t[1] = Sin(theta);
                t[5] = Cos(theta);
                t[10] = 1;
                t[15] = 1;
        } else if (y == 1) {
                t[0] = Cos(theta);
                t[2] = -Sin(theta);
                t[8] = Sin(theta);
                t[10] = Cos(theta);
                t[5] = 1;
                t[15] = 1;
        } else if (x == 1) {
                t[5] = Cos(theta);
                t[6] = Sin(theta);
                t[9] = -Sin(theta);
                t[10] = Cos(theta);
                t[0] = 1;
                t[15] = 1;
        }

        glMultMatrixf(t);

}

/*
 * Constructs a scaling matrix from the given input parameters and multiply
 * the current geometry matrix by that scaling matrix.
 *
 * x,y,z - values to scale in the x, y, and z directions
 */
void glScalef(GLfloat x, GLfloat y, GLfloat z) {
	fixed *m = cur_matrix;
	fixed t[16];
	
	if (m == NULL)
		return;
	
	InitializeMatrix(t);
	t[0] = x;
	t[5] = y;
	t[10] = z;

	glMultMatrixf(t);
}

/* 
 * Translate the origin by x,y,z by constructing a translation 4x4 matrix and
 * multiplying the current geometry matrix by that new matrix.
 *
 * x, y, z - units to translate in 3D world space
 */
void glTranslatef(GLfloat x, GLfloat y, GLfloat z) {
	fixed *m = cur_matrix;
	const fixed fx = DTF(x);
	const fixed fy = DTF(y);
	const fixed fz = DTF(z);
	
	if (m == NULL)
		return;
	
	/** Optimized for speed */
	/** 4x4 matrix multiplication */
	m[0] += fixmul(fx, m[3]);
	m[1] += fixmul(fy, m[3]);
	m[2] += fixmul(fz, m[3]);
	m[4] += fixmul(fx, m[7]);
	m[5] += fixmul(fy, m[7]);
	m[6] += fixmul(fz, m[7]);
	m[8] += fixmul(fx, m[11]);
	m[9] += fixmul(fy, m[11]);
	m[10] += fixmul(fz, m[11]);
	m[12] += fixmul(fx, m[15]);
	m[13] += fixmul(fy, m[15]);
	m[14] += fixmul(fz, m[15]);
}

void glPopMatrix() {
	switch (matrix_mode) {
		case GL_MODELVIEW:
			/** Error to pop an empty stack */
			if (modv_level <= 0)
				return;
			modv_level--;
			//cur_matrix = modv_matrix[modv_level];
			break;
		case GL_PROJECTION:
			/** Error to pop an empty stack */
			if (proj_level <= 0)
				return;
			proj_level--;
			//cur_matrix = proj_matrix[proj_level];
			break;
	}
}

static void CopyMatrixFTF(const fixed* src, fixed* dst) {
	char i = 0;
	for (; i < 16; i++)
		dst[i] = src[i];
}

void glPushMatrix() {
	switch (matrix_mode) {
		case GL_MODELVIEW:
			/** Error to pop an empty stack */
			if (modv_level >= (MAX_MAT_STACK_MODV - 1))
				return;
			CopyMatrixFTF(modv_matrix[modv_level],
				modv_matrix[modv_level + 1]);
			modv_level++;
			//cur_matrix = modv_matrix[modv_level];
			break;
		case GL_PROJECTION:
			/** Error to pop an empty stack */
			if (proj_level >= (MAX_MAT_STACK_PROJ - 1))
				return;
			CopyMatrixFTF(proj_matrix[proj_level],
				proj_matrix[proj_level + 1]);
			proj_level++;
			//cur_matrix = proj_matrix[proj_level];
			break;
	}
}

void glMatrixMode(GLenum mode) {
	matrix_mode = mode;
	switch (matrix_mode) {
		case GL_MODELVIEW:
			//cur_matrix = modv_matrix[modv_level];
			break;
		case GL_PROJECTION:
			//cur_matrix = proj_matrix[proj_level];
			break;
		default:
			//cur_matrix = NULL;
	}
}

void glNormal3f(GLfloat nx, GLfloat ny, GLfloat nz) {
	cur_normal[0] = nx;
	cur_normal[1] = ny;
	cur_normal[2] = nz;
	cur_normal[3] = 1;
}

void glEnable(GLenum cap) {
	if (cap >= GL_LIGHT0 && cap <= GL_LIGHT7) {
		lights[(cap - GL_LIGHT0)].enabled = 1;
	}

	switch (cap) {
		case GL_CULL_FACE:
			culling = 1;
			break;
		case GL_LIGHTING:
			lighting = 1;
			break;
		default:
			break;
	}
}

void glDisable(GLenum cap) {
	if (cap >= GL_LIGHT0 && cap <= GL_LIGHT7) {
		lights[(cap - GL_LIGHT0)].enabled = 0;
	}

	switch (cap) {
		case GL_CULL_FACE:
			culling = 0;
			break;
		case GL_LIGHTING:
			lighting = 0;
			break;
		default:
			break;
	}
}

/* Sets an attribute of a light.  The light is identified by the GLenum light,
 * the property is selected by pname, and the values set by params.
 */
void glLightfv(GLenum light, GLenum pname, const GLfloat *params) {
	int i;

	if (params == NULL)
		return;

	if (light > GL_LIGHT7 || light < GL_LIGHT0)
		return;

	switch(pname) {
		case GL_AMBIENT:
			for (i=0;i<4;i++) {
				lights[light-GL_LIGHT0].ambient[i] = params[i];
			}
			break;
		case GL_DIFFUSE:
			for (i=0;i<4;i++) {
				lights[light-GL_LIGHT0].diffuse[i] = params[i];
			}
			break;
		case GL_SPECULAR:
			for (i=0;i<4;i++) {
				lights[light-GL_LIGHT0].specular[i]= params[i];
			}
			break;
		case GL_POSITION:
			for (i=0;i<4;i++) {
				lights[light-GL_LIGHT0].position[i]= params[i];
			}
			break;
		default:
			break;
	}
}

/* Sets an attribute of a light.  The light is identified by the GLenum light,
 * the property is selected by pname, and the values set by params.
 */
void glGetLightfv(GLenum light, GLenum pname, GLfloat *params) {
	int i;

	if (params == NULL)
		return;

	if (light > GL_LIGHT7 || light < GL_LIGHT0)
		return;

	switch(pname) {
		case GL_AMBIENT:
			for (i=0;i<4;i++) {
				params[i] = lights[light-GL_LIGHT0].ambient[i];
			}
			break;
		case GL_DIFFUSE:
			for (i=0;i<4;i++) {
				params[i] = lights[light-GL_LIGHT0].diffuse[i];
			}
			break;
		case GL_SPECULAR:
			for (i=0;i<4;i++) {
				params[i] = lights[light-GL_LIGHT0].specular[i];
			}
			break;
		case GL_POSITION:
			for (i=0;i<4;i++) {
				params[i] = lights[light-GL_LIGHT0].position[i];
			}
			break;
		default:
			break;
	}
}

/*
 *
 */
void glPolygonMode(GLenum face, GLenum mode) {
	if (mode == GL_FILL)
		wireframe = 0;
	else if (mode == GL_LINE)
		wireframe = 1;
}

/* 
 * Sets the drawing area on the Palm screen.  Basically sets up a window into
 * which all gl primitives will be rendered according to the perspective rules
 * set up with gluPerspective(...) or glOrtho(...).
 *
 * x1, y1 - (int, int) coordinate of top left corner
 * x2, y2 - (int, int) coordinate of bottom right corner
 */
void WinSetGLArea(int x1, int y1, int x2, int y2) {
	int i;
	Err err;

	screen_width = x2 - x1;
	screen_height = y2 - y1;
	screen_startx = x1;
	screen_starty = y1;

	for (i=0;i<8;i++) {
		lights[i].enabled = 0;
	}

	fgChangeToGreyscale();

	/** Create the second buffer if it has not been created yet */
	if (two_created == 0) {

		//ZBufferInit();

		/** create offscreen bitmap */
		one = WinGetActiveWindow();
		two = WinCreateOffscreenWindow(screen_width, 
				screen_height, screenFormat, &err);
		one = WinSetDrawWindow(two);
		WinSetActiveWindow(one);
		two_created = 1;
	}
}

void glMap2f( GLenum target,
                        GLfloat u1, GLfloat u2, GLint ustride, GLint uorder,
                        GLfloat v1, GLfloat v2, GLint vstride, GLint vorder,
                        const GLfloat *points ) {
	// nothing here yet.
}

void glMapGrid2f(GLint un, GLfloat u1, GLfloat u2,
                        GLint vn, GLfloat v1, GLfloat v2 ) {
	// nothing here yet.
}

void glEvalMesh2(GLenum mode, GLint i1, GLint i2, GLint j1, GLint j2) {
	// nothing here yet.
}

void glPushAttrib(GLbitfield mask) {
	// nothing here yet.
}

void glPopAttrib() {
	// nothing here yet.
}

void glDrawPixels(GLsizei width, GLsizei height, GLenum format,
                        GLenum type, const GLvoid *pixels) {
	// nothing here yet.
}

void glBitmap(GLsizei width, GLsizei height, GLfloat xbo,
                        GLfloat ybo, GLfloat xbi, GLfloat ybi,
                        const GLubyte *bitmap) {
	// nothing here yet.
}

/*
 * Set back to normal mono screen rendering mode
 */
void WinRemoveGLArea() {
	fgChangeFromGreyscale();
}

/*** Research Line *** DO NOT CROSS *** Research Line *** DO NOT CROSS ***/

/*
 * There's nothing to see here.  Move along.
 */

/*
 * End of minigl_main.c
 *
 * References:  
 *   Foley, et al.  Computer Graphics.  Addison Wesley, 1990.
 *   Woo, et al.  OpenGL Programming Reference.  Addison Wesley, 1997.
 */

