/*
 * fastmath.h
 * fast math functions for miniGL
 * Michael Sherman, Digital Sandbox, Inc.
 *
 * Code borrowed from Rick Huebner, MathLib
 */

#include "fastmath.h"


#define __HI32(x) *((u_int32_t *) &x)
#define __LO32(x) *((u_int32_t *) &x + 1)

/* Get two 32 bit ints from a double.  */
#define EXTRACT_WORDS(ix0,ix1,d) \
do {                             \
    (ix0) = __HI32(d);       \
    (ix1) = __LO32(d);       \
} while (0)  

/* Get the more significant 32 bit int from a double.  */
#define GET_HIGH_WORD(i,d) \
do {                       \
    (i) = __HI32(d);   \
} while (0)

/* Get the less significant 32 bit int from a double.  */
#define GET_LOW_WORD(i,d) \
do {                      \
    (i) = __LO32(d);  \
} while (0)

/* Set a double from two 32 bit ints.  */
#define INSERT_WORDS(d,ix0,ix1) \
do {                            \
    __HI32(d) = (ix0);      \
    __LO32(d) = (ix1);      \
} while (0) 

/* Set the more significant 32 bits of a double from an int.  */
#define SET_HIGH_WORD(d,v)\
do {                      \
    __HI32(d) = (v);  \
} while (0)

/* Set the less significant 32 bits of a double from an int.  */
#define SET_LOW_WORD(d,v) \
do {                     \
    __LO32(d) = (v);  \
} while (0)          


static int __kernel_rem_pio2(double *x, double *y, int e0, int nx, int prec, 
		const int32_t *ipio2);
static int32_t __ieee754_rem_pio2(double x, double *y);
static double __kernel_cos(double x, double y);
static double __kernel_sin(double x, double y, int iy);
static double __fabs(double x);
static double __scalbn (double x, int n);
static double __floor(double x);
static double __copysign(double x, double y);


static const double one = 1.0, tiny=1.0e-300; 

/*
 * From  MathLib, e_sqrt.c 
 * Version 1.0, 15 August 1997, Rick Huebner
 */
double Sqrt(double x) {
    double z;
    const int32_t sign = (int)0x80000000;
    int32_t ix0,s0,q,m,t,i;
    u_int32_t r,t1,s1,ix1,q1;

    EXTRACT_WORDS(ix0,ix1,x);   

    /* take care of Inf and NaN */
    if((ix0&0x7ff00000)==0x7ff00000) {
		return x*x+x;               /* sqrt(NaN)=NaN, sqrt(+inf)=+inf
                                           sqrt(-inf)=sNaN */
    }
    /* take care of zero */
    if(ix0<=0) {
		if(((ix0&(~sign))|ix1)==0) return x;/* sqrt(+-0) = +-0 */
		else if(ix0<0)
			return (x-x)/(x-x);             /* sqrt(-ve) = sNaN */
    }
    /* normalize x */
    m = (ix0>>20);
    if(m==0) {                              /* subnormal x */
		while(ix0==0) {
			m -= 21;
			ix0 |= (ix1>>11); ix1 <<= 21;
		}
		for(i=0;(ix0&0x00100000)==0;i++) ix0<<=1;
		m -= i-1;
		ix0 |= (ix1>>(32-i));                                      
		ix1 <<= i;
    }
    m -= 1023;      /* unbias exponent */
    ix0 = (ix0&0x000fffff)|0x00100000;
    if(m&1){        /* odd m, double x to make it even */
		ix0 += ix0 + ((ix1&sign)>>31);
		ix1 += ix1;
    }
    m >>= 1;        /* m = [m/2] */

    /* generate sqrt(x) bit by bit */
    ix0 += ix0 + ((ix1&sign)>>31);
    ix1 += ix1;
    q = q1 = s0 = s1 = 0;   /* [q,q1] = sqrt(x) */
    r = 0x00200000;         /* r = moving bit from right to left */

    while(r!=0) {
		t = s0+r;
		if(t<=ix0) {
			s0   = t+r;
			ix0 -= t;                       
			q   += r;
		}
		ix0 += ix0 + ((ix1&sign)>>31);
		ix1 += ix1;
		r>>=1;
    }

    r = sign;
    while(r!=0) {
		t1 = s1+r;
		t  = s0;
		if((t<ix0)||((t==ix0)&&(t1<=ix1))) {
			s1  = t1+r;
			if(((t1&sign)==sign)&&(s1&sign)==0) s0 += 1;
			ix0 -= t;
			if (ix1 < t1) ix0 -= 1;
			ix1 -= t1;
			q1  += r;
		}
		ix0 += ix0 + ((ix1&sign)>>31);
		ix1 += ix1;
		r>>=1;                            
    }

    /* use floating add to find out rounding direction */
    if((ix0|ix1)!=0) {
		z = one-tiny; /* trigger inexact flag */
		if (z>=one) {
			z = one+tiny;
			if (q1==(u_int32_t)0xffffffff) { q1=0; q += 1;}
			else if (z>one) {
				if (q1==(u_int32_t)0xfffffffe) q+=1;
				q1+=2;
			} else
				q1 += (q1&1);
		}
    }
    ix0 = (q>>1)+0x3fe00000;
    ix1 =  q1>>1;
    if ((q&1)==1) ix1 |= sign;
    ix0 += (m <<20);
    INSERT_WORDS(z,ix0,ix1);

    return z;               
}

/*
 * ====================================================
 * Copyright (C) 1993 by Sun Microsystems, Inc. All rights reserved.
 *
 * Developed at SunPro, a Sun Microsystems, Inc. business.
 * Permission to use, copy, modify, and distribute this
 * software is freely granted, provided that this notice
 * is preserved.
 * ====================================================
 */       

static const double         
half =  5.00000000000000000000e-01, /* 0x3FE00000, 0x00000000 */
    S1  = -1.66666666666666324348e-01, /* 0xBFC55555, 0x55555549 */
    S2  =  8.33333333332248946124e-03, /* 0x3F811111, 0x1110F8A6 */
    S3  = -1.98412698298579493134e-04, /* 0xBF2A01A0, 0x19C161D5 */
    S4  =  2.75573137070700676789e-06, /* 0x3EC71DE3, 0x57B1FE7D */
    S5  = -2.50507602534068634195e-08, /* 0xBE5AE5E6, 0x8A2B9CEB */
    S6  =  1.58969099521155010221e-10; /* 0x3DE5D93A, 0x5ACFD57C */   

static const double    
ONE =  1.00000000000000000000e+00, /* 0x3FF00000, 0x00000000 */
    C1  =  4.16666666666666019037e-02, /* 0x3FA55555, 0x5555554C */
    C2  = -1.38888888888741095749e-03, /* 0xBF56C16C, 0x16C15177 */
    C3  =  2.48015872894767294178e-05, /* 0x3EFA01A0, 0x19CB1590 */
    C4  = -2.75573143513906633035e-07, /* 0xBE927E4F, 0x809C52AD */
    C5  =  2.08757232129817482790e-09, /* 0x3E21EE9E, 0xBDB4B1C4 */
    C6  = -1.13596475577881948265e-11; /* 0xBDA8FAE9, 0xBE8838D4 */  

static double __kernel_sin(double x, double y, int iy) {
    double z,r,v;
    int32_t ix;
    GET_HIGH_WORD(ix,x);
    ix &= 0x7fffffff;                       /* high word of x */
    if(ix<0x3e400000)                       /* |x| < 2**-27 */
    {if((int)x==0) return x;}            /* generate inexact */
    z       =  x*x;
    v       =  z*x;
    r       =  S2+z*(S3+z*(S4+z*(S5+z*S6)));
    if(iy==0) return x+v*(S1+z*r);
    else      return x-((z*(half*y-v*r)-y)-v*S1);
}   

static double __kernel_cos(double x, double y)  {
    double a,hz,z,r,qx;
    int32_t ix;
    GET_HIGH_WORD(ix,x);
    ix &= 0x7fffffff;                       /* ix = |x|'s high word*/
    if(ix<0x3e400000) {                     /* if x < 2**27 */
		if(((int)x)==0) return ONE;         /* generate inexact */
    }
    z  = x*x;
    r  = z*(C1+z*(C2+z*(C3+z*(C4+z*(C5+z*C6)))));
    if(ix < 0x3FD33333)                     /* if |x| < 0.3 */
		return ONE - (0.5*z - (z*r - x*y));
    else {
		if(ix > 0x3fe90000) {               /* x > 0.78125 */
			qx = 0.28125;
		} else {
			INSERT_WORDS(qx,ix-0x00200000,0);       /* x/4 */
		}
		hz = 0.5*z-qx;
		a  = ONE-qx;
		return a - (hz - (z*r-x*y));
    }
}        

static const float sin_tab[] = {
	0.000000, 0.012272, 0.024541, 0.036807,
	0.049068, 0.061321, 0.073565, 0.085797,
	0.098017, 0.110222, 0.122411, 0.134581,
	0.146730, 0.158858, 0.170962, 0.183040,
	0.195090, 0.207111, 0.219101, 0.231058,
	0.242980, 0.254866, 0.266713, 0.278520,
	0.290285, 0.302006, 0.313682, 0.325310,
	0.336890, 0.348419, 0.359895, 0.371317,
	0.382683, 0.393992, 0.405241, 0.416430,
	0.427555, 0.438616, 0.449611, 0.460539,
	0.471397, 0.482184, 0.492898, 0.503538,
	0.514103, 0.524590, 0.534998, 0.545325,
	0.555570, 0.565732, 0.575808, 0.585798,
	0.595699, 0.605511, 0.615232, 0.624860,
	0.634393, 0.643832, 0.653173, 0.662416,
	0.671559, 0.680601, 0.689541, 0.698376,
	0.707107, 0.715731, 0.724247, 0.732654,
	0.740951, 0.749136, 0.757209, 0.765167,
	0.773010, 0.780737, 0.788346, 0.795837,
	0.803208, 0.810457, 0.817585, 0.824589,
	0.831470, 0.838225, 0.844854, 0.851355,
	0.857729, 0.863973, 0.870087, 0.876070,
	0.881921, 0.887640, 0.893224, 0.898674,
	0.903989, 0.909168, 0.914210, 0.919114,
	0.923880, 0.928506, 0.932993, 0.937339,
	0.941544, 0.945607, 0.949528, 0.953306,
	0.956940, 0.960431, 0.963776, 0.966976,
	0.970031, 0.972940, 0.975702, 0.978317,
	0.980785, 0.983105, 0.985278, 0.987301,
	0.989177, 0.990903, 0.992480, 0.993907,
	0.995185, 0.996313, 0.997290, 0.998118,
	0.998795, 0.999322, 0.999699, 0.999925
};

static const float PIx2 = 3.14159265*2;

float Sin(float x) {
	const float percent = x/PIx2;
	const long idx_untrimmed = (percent+200)*512;
	const int idx = idx_untrimmed % 512;
	if (idx < 128)
		return sin_tab[idx];
	else if (idx < 256)
		return sin_tab[256 - idx];
	else if (idx < 384)
		return -sin_tab[idx - 256];
	else
		return -sin_tab[512 - idx];
}

float Cos(float x) {
	const float percent = x/PIx2;
	const long idx_untrimmed = (percent+200)*512;
	const int idx = idx_untrimmed % 512;
	if (idx < 128)
		return sin_tab[128 - idx];
	else if (idx < 256)
		return -sin_tab[idx - 128];
	else if (idx < 384)
		return -sin_tab[384 - idx];
	else
		return sin_tab[idx - 384];
}


static const double    
zero =  0.00000000000000000000e+00, /* 0x00000000, 0x00000000 */
    two24 =  1.67772160000000000000e+07, /* 0x41700000, 0x00000000 */
    invpio2 =  6.36619772367581382433e-01, /* 0x3FE45F30, 0x6DC9C883 */
    pio2_1  =  1.57079632673412561417e+00, /* 0x3FF921FB, 0x54400000 */
    pio2_1t =  6.07710050650619224932e-11, /* 0x3DD0B461, 0x1A626331 */
    pio2_2  =  6.07710050630396597660e-11, /* 0x3DD0B461, 0x1A600000 */
    pio2_2t =  2.02226624879595063154e-21, /* 0x3BA3198A, 0x2E037073 */
    pio2_3  =  2.02226624871116645580e-21, /* 0x3BA3198A, 0x2E000000 */
    pio2_3t =  8.47842766036889956997e-32; /* 0x397B839A, 0x252049C1 */  

/*
 * Table of constants for 2/pi, 396 Hex digits (476 decimal) of 2/pi
 */
static const int32_t two_over_pi[] = {
    0xA2F983, 0x6E4E44, 0x1529FC, 0x2757D1, 
    0xF534DD, 0xC0DB62, 0x95993C, 0x439041,
    0xFE5163, 0xABDEBB, 0xC561B7, 0x246E3A,
    0x424DD2, 0xE00649, 0x2EEA09, 0xD1921C,
    0xFE1DEB, 0x1CB129, 0xA73EE8, 0x8235F5,
    0x2EBB44, 0x84E99C, 0x7026B4, 0x5F7E41,
    0x3991D6, 0x398353, 0x39F49C, 0x845F8B,
    0xBDF928, 0x3B1FF8, 0x97FFDE, 0x05980F,
    0xEF2F11, 0x8B5A0A, 0x6D1F6D, 0x367ECF,
    0x27CB09, 0xB74F46, 0x3F669E, 0x5FEA2D,
    0x7527BA, 0xC7EBE5, 0xF17B3D, 0x0739F7,
    0x8A5292, 0xEA6BFB, 0x5FB11F, 0x8D5D08, 
    0x560330, 0x46FC7B, 0x6BABF0, 0xCFBC20,
    0x9AF436, 0x1DA9E3, 0x91615E, 0xE61B08,
    0x659985, 0x5F14A0, 0x68408D, 0xFFD880,
    0x4D7327, 0x310606, 0x1556CA, 0x73A8C9,
    0x60E27B, 0xC08C6B
};                                   

static const int32_t npio2_hw[] = {
	0x3FF921FB, 0x400921FB, 0x4012D97C, 0x401921FB,
	0x401F6A7A, 0x4022D97C, 0x4025FDBB, 0x402921FB,
	0x402C463A, 0x402F6A7A, 0x4031475C, 0x4032D97C,
	0x40346B9C, 0x4035FDBB, 0x40378FDB, 0x403921FB,
	0x403AB41B, 0x403C463A, 0x403DD85A, 0x403F6A7A,
	0x40407E4C, 0x4041475C, 0x4042106C, 0x4042D97C,
	0x4043A28C, 0x40446B9C, 0x404534AC, 0x4045FDBB,
	0x4046C6CB, 0x40478FDB, 0x404858EB, 0x404921FB
};

static int32_t __ieee754_rem_pio2(double x, double *y) {
    double z,w,t,r,fn;
    double tx[3];
    int32_t e0,i,j,nx,n,ix,hx;
    u_int32_t low;

    GET_HIGH_WORD(hx,x);            /* high word of x */
    ix = hx&0x7fffffff;
    if(ix<=0x3fe921fb)   /* |x| ~<= pi/4 , no need for reduction */
    {y[0] = x; y[1] = 0; return 0;}
    if(ix<0x4002d97c) {  /* |x| < 3pi/4, special case with n=+-1 */
		if(hx>0) {
			z = x - pio2_1;
			if(ix!=0x3ff921fb) {    /* 33+53 bit pi is good enough */
				y[0] = z - pio2_1t;
				y[1] = (z-y[0])-pio2_1t;
			} else {                /* near pi/2, use 33+33+53 bit pi */
				z -= pio2_2;
				y[0] = z - pio2_2t;
				y[1] = (z-y[0])-pio2_2t;
			}
			return 1;
		} else {    /* negative x */
			z = x + pio2_1;
			if(ix!=0x3ff921fb) {    /* 33+53 bit pi is good enough */
				y[0] = z + pio2_1t;
				y[1] = (z-y[0])+pio2_1t;
			} else {                /* near pi/2, use 33+33+53 bit pi */
				z += pio2_2;
				y[0] = z + pio2_2t;
				y[1] = (z-y[0])+pio2_2t;
			}
			return -1;
		}
    }                                
    if(ix<=0x413921fb) { /* |x| ~<= 2^19*(pi/2), medium size */
		t  = __fabs(x);
		n  = (int32_t) (t*invpio2+half);
		fn = (double)n;
		r  = t-fn*pio2_1;
		w  = fn*pio2_1t;    /* 1st round good to 85 bit */
		if(n<32&&ix!=npio2_hw[n-1]) {
			y[0] = r-w;     /* quick check no cancellation */
		} else {
			u_int32_t high;
			j  = ix>>20;
			y[0] = r-w;
			GET_HIGH_WORD(high,y[0]);
			i = j-((high>>20)&0x7ff);
			if(i>16) {  /* 2nd iteration needed, good to 118 */
				t  = r;
				w  = fn*pio2_2;
				r  = t-w;
				w  = fn*pio2_2t-((t-r)-w);
				y[0] = r-w;
				GET_HIGH_WORD(high,y[0]);
				i = j-((high>>20)&0x7ff);
				if(i>49)  { /* 3rd iteration need, 151 bits acc */
					t  = r; /* will cover all possible cases */
					w  = fn*pio2_3;
					r  = t-w;
					w  = fn*pio2_3t-((t-r)-w);
					y[0] = r-w;
				}
			}
		}
		y[1] = (r-y[0])-w;
		if(hx<0)    {y[0] = -y[0]; y[1] = -y[1]; return -n;}
		else         return n;
    }                                             
    /*
     * all other (large) arguments
     */
    if(ix>=0x7ff00000) {            /* x is inf or NaN */
		y[0]=y[1]=x-x; return 0;
    }
    /* set z = scalbn(|x|,ilogb(x)-23) */
    GET_LOW_WORD(low,x);
    SET_LOW_WORD(z,low);
    e0      = (ix>>20)-1046;        /* e0 = ilogb(z)-23; */
    SET_HIGH_WORD(z, ix - ((int32_t)(e0<<20)));
    for(i=0;i<2;i++) {
		tx[i] = (double)((int32_t)(z));
		z     = (z-tx[i])*two24;
    }
    tx[2] = z;
    nx = 3;
    while(tx[nx-1]==zero) nx--;     /* skip zero term */
    n  =  __kernel_rem_pio2(tx,y,e0,nx,2,two_over_pi);
    if(hx<0) {y[0] = -y[0]; y[1] = -y[1]; return -n;}
    return n;              
}

static const double     
//two24   =  1.67772160000000000000e+07, /* 0x41700000, 0x00000000 */
twon24  =  5.96046447753906250000e-08; /* 0x3E700000, 0x00000000 */


static const double PIo2[] = {
	1.57079625129699707031e+00, /* 0x3FF921FB, 0x40000000 */
	7.54978941586159635335e-08, /* 0x3E74442D, 0x00000000 */
	5.39030252995776476554e-15, /* 0x3CF84698, 0x80000000 */
	3.28200341580791294123e-22, /* 0x3B78CC51, 0x60000000 */
	1.27065575308067607349e-29, /* 0x39F01B83, 0x80000000 */
	1.22933308981111328932e-36, /* 0x387A2520, 0x40000000 */
	2.73370053816464559624e-44, /* 0x36E38222, 0x80000000 */
	2.16741683877804819444e-51 /* 0x3569F31D, 0x00000000 */
};

static const int init_jk[] = {
	2, 3, 4, 6
};

static int __kernel_rem_pio2(double *x, double *y, int e0, int nx, int prec,
        const int32_t *ipio2) {
    int32_t jz,carry,n,iq[20],i,j,k,m,q0,ih;
    double z,fw,f[20],fq[20],q[20];

    /* initialize jk*/
    const int32_t jk = init_jk[prec];
    const int32_t jp = jk;

    /* determine jx,jv,q0, note that 3>q0 */
    const int32_t jx =  nx-1;
    int32_t jv = (e0-3)/24;
	if(jv<0) jv=0;
    q0 =  e0-24*(jv+1);

    /* set up f[0] to f[jx+jk] where f[jx+jk] = ipio2[jv+jk] */
    j = jv-jx; m = jx+jk;
    for(i=0;i<=m;i++,j++) f[i] = (j<0)? zero : (double) ipio2[j];

    /* compute q[0],q[1],...q[jk] */
    for (i=0;i<=jk;i++) {
		for(j=0,fw=0.0;j<=jx;j++) fw += x[j]*f[jx+i-j]; q[i] = fw;
    }

    jz = jk;
recompute:
    /* distill q[] into iq[] reversingly */
    for(i=0,j=jz,z=q[jz];j>0;i++,j--) {
		fw    =  (double)((int32_t)(twon24* z));
		iq[i] =  (int32_t)(z-two24*fw);
		z     =  q[j-1]+fw;
    }

    /* compute n */
    z  = __scalbn(z,q0);            /* actual value of z */
    z -= 8.0*__floor(z*0.125);              /* trim off integer >= 8 */
    n  = (int32_t) z;
    z -= (double)n;
    ih = 0;
    if(q0>0) {      /* need iq[jz-1] to determine n */
		i  = (iq[jz-1]>>(24-q0)); n += i;
		iq[jz-1] -= i<<(24-q0);               
		ih = iq[jz-1]>>(23-q0);
    }
    else if(q0==0) ih = iq[jz-1]>>23;
    else if(z>=0.5) ih=2;

    if(ih>0) {      /* q > 0.5 */
		n += 1; carry = 0;
		for(i=0;i<jz ;i++) {        /* compute 1-q */
			j = iq[i];
			if(carry==0) {
				if(j!=0) {
					carry = 1; iq[i] = 0x1000000- j;
				}
			} else  iq[i] = 0xffffff - j;
		}
		if(q0>0) {          /* rare case: chance is 1 in 12 */
			switch(q0) {
                case 1:
					iq[jz-1] &= 0x7fffff; break;
                case 2:
					iq[jz-1] &= 0x3fffff; break;
			}
		}
		if(ih==2) {
			z = one - z;
			if(carry!=0) z -= __scalbn(one,q0);
		}
    }

    /* check if recomputation is needed */
    if(z==zero) {
		j = 0;
		for (i=jz-1;i>=jk;i--) j |= iq[i];
		if(j==0) { /* need recomputation */
			for(k=1;iq[jk-k]==0;k++);   /* k = no. of terms needed */

			for(i=jz+1;i<=jz+k;i++) {   /* add q[jz+1] to q[jz+k] */
				f[jx+i] = (double) ipio2[jv+i];
				for(j=0,fw=0.0;j<=jx;j++) fw += x[j]*f[jx+i-j];
				q[i] = fw;
			}
			jz += k;
			goto recompute;
		}
    }                                 

    /* chop off zero terms */
    if(z==0.0) {
		jz -= 1; q0 -= 24;
		while(iq[jz]==0) { jz--; q0-=24;}
    } else { /* break z into 24-bit if necessary */
		z = __scalbn(z,-q0);
		if(z>=two24) {
			fw = (double)((int32_t)(twon24*z));
			iq[jz] = (int32_t)(z-two24*fw);
			jz += 1; q0 += 24;
			iq[jz] = (int32_t) fw;
		} else iq[jz] = (int32_t) z ;
    }

    /* convert integer "bit" chunk to floating-point value */
    fw = __scalbn(one,q0);
    for(i=jz;i>=0;i--) {
		q[i] = fw*(double)iq[i]; fw*=twon24;
    }

    /* compute PIo2[0,...,jp]*q[jz,...,0] */
    for(i=jz;i>=0;i--) {
		for(fw=0.0,k=0;k<=jp&&k<=jz-i;k++) fw += PIo2[k]*q[i+k];
		fq[jz-i] = fw;
    }                                      

    /* compress fq[] into y[] */
    switch(prec) {
		case 0:
			fw = 0.0;
			for (i=jz;i>=0;i--) fw += fq[i];
			y[0] = (ih==0)? fw: -fw;
			break;
		case 1:
		case 2:
			fw = 0.0;
			for (i=jz;i>=0;i--) fw += fq[i];
			y[0] = (ih==0)? fw: -fw;
			fw = fq[0]-fw;
			for (i=1;i<=jz;i++) fw += fq[i];
			y[1] = (ih==0)? fw: -fw;
			break;
		case 3:     /* painful */
			for (i=jz;i>0;i--) {
				fw      = fq[i-1]+fq[i];
				fq[i]  += fq[i-1]-fw;
				fq[i-1] = fw;
			}
			for (i=jz;i>1;i--) {
				fw      = fq[i-1]+fq[i];
				fq[i]  += fq[i-1]-fw;
				fq[i-1] = fw;
			}
			for (fw=0.0,i=jz;i>=2;i--) fw += fq[i];
			if(ih==0) {
				y[0] =  fq[0]; y[1] =  fq[1]; y[2] =  fw;
			} else {
				y[0] = -fq[0]; y[1] = -fq[1]; y[2] = -fw;
			}
    }
    return n&7;                 
}

static double __fabs(double x) {
    u_int32_t high;
    GET_HIGH_WORD(high,x);
    SET_HIGH_WORD(x,high&0x7fffffff);
    return x;
}    


static const double   
two54   =  1.80143985094819840000e+16, /* 0x43500000, 0x00000000 */
    twom54  =  5.55111512312578270212e-17, /* 0x3C900000, 0x00000000 */
    huge   = 1.0e+300;

static double __scalbn (double x, int n) {
    int32_t k,hx,lx;
    EXTRACT_WORDS(hx,lx,x);
    k = (hx&0x7ff00000)>>20;                /* extract exponent */
    if (k==0) {                             /* 0 or subnormal x */
		if ((lx|(hx&0x7fffffff))==0) return x; /* +-0 */
		x *= two54;
		GET_HIGH_WORD(hx,x);
		k = ((hx&0x7ff00000)>>20) - 54;
		if (n< -50000) return tiny*x;       /*underflow*/
    }
    if (k==0x7ff) return x+x;               /* NaN or Inf */
    k = k+n;
    if (k >  0x7fe) return huge*__copysign(huge,x); /* overflow  */
    if (k > 0)                              /* normal result */
    {SET_HIGH_WORD(x,(hx&0x800fffff)|(k<<20)); return x;}
    if (k <= -54)
		if (n > 50000)      /* in case integer overflow in n+k */
			return huge*__copysign(huge,x); /*overflow*/
		else return tiny*__copysign(tiny,x);        /*underflow*/
    k += 54;                                /* subnormal result */
    SET_HIGH_WORD(x,(hx&0x800fffff)|(k<<20));
    return x*twom54;
}                          

static double __floor(double x) {
    int32_t i0,i1,j0;
    u_int32_t i,j;
    EXTRACT_WORDS(i0,i1,x);
    j0 = ((i0>>20)&0x7ff)-0x3ff;
    if(j0<20) {
		if(j0<0) {  /* raise inexact if x != 0 */
			if(huge+x>0.0) {/* return 0*sign(x) if |x|<1 */
				if(i0>=0) {i0=i1=0;}
				else if(((i0&0x7fffffff)|i1)!=0)
				{ i0=0xbff00000;i1=0;}
			}
		} else {
			i = (0x000fffff)>>j0;
			if(((i0&i)|i1)==0) return x; /* x is integral */
			if(huge+x>0.0) {        /* raise inexact flag */
				if(i0<0) i0 += (0x00100000)>>j0;
				i0 &= (~i); i1=0;
			}
		}
    } else if (j0>51) {
		if(j0==0x400) return x+x;   /* inf or NaN */
		else return x;              /* x is integral */
    } else {
		i = ((u_int32_t)(0xffffffff))>>(j0-20);
		if((i1&i)==0) return x;     /* x is integral */
		if(huge+x>0.0) {            /* raise inexact flag */
			if(i0<0) {
				if(j0==20) i0+=1;
				else {
					j = i1+(1<<(52-j0));
					if(j<i1) i0 +=1 ;       /* got a carry */
					i1=j;
				}
			}
			i1 &= (~i);
		}
    }
    INSERT_WORDS(x,i0,i1);
    return x;
}                                  

static double __copysign(double x, double y) {
    u_int32_t hx,hy;
    GET_HIGH_WORD(hx,x);
    GET_HIGH_WORD(hy,y);
    SET_HIGH_WORD(x,(hx&0x7fffffff)|(hy&0x80000000));
    return x;
}   

double Atan(double x) {
    double atanhi[4], atanlo[4], aT[11];
    double w,s1,s2,z;
    int32_t ix,hx,id;

    atanhi[0] = 4.63647609000806093515e-01; 
                        /* atan(0.5)hi 0x3FDDAC67, 0x0561BB4F */
    atanhi[1] = 7.85398163397448278999e-01;
                        /* atan(1.0)hi 0x3FE921FB, 0x54442D18 */
    atanhi[2] = 9.82793723247329054082e-01;
                        /* atan(1.5)hi 0x3FEF730B, 0xD281F69B */
    atanhi[3] = 1.57079632679489655800e+00;
                        /* atan(inf)hi 0x3FF921FB, 0x54442D18 */
    atanlo[0] = 2.26987774529616870924e-17; 
                        /* atan(0.5)lo 0x3C7A2B7F, 0x222F65E2 */
    atanlo[1] = 3.06161699786838301793e-17; 
                        /* atan(1.0)lo 0x3C81A626, 0x33145C07 */
    atanlo[2] = 1.39033110312309984516e-17; 
                        /* atan(1.5)lo 0x3C700788, 0x7AF0CBBD */
    atanlo[3] = 6.12323399573676603587e-17; 
                        /* atan(inf)lo 0x3C91A626, 0x33145C07 */
    aT[0] =  3.33333333333329318027e-01; /* 0x3FD55555, 0x5555550D */
    aT[1] = -1.99999999998764832476e-01; /* 0xBFC99999, 0x9998EBC4 */
    aT[2] =  1.42857142725034663711e-01; /* 0x3FC24924, 0x920083FF */
    aT[3] = -1.11111104054623557880e-01; /* 0xBFBC71C6, 0xFE231671 */
    aT[4] =  9.09088713343650656196e-02; /* 0x3FB745CD, 0xC54C206E */
    aT[5] = -7.69187620504482999495e-02; /* 0xBFB3B0F2, 0xAF749A6D */
    aT[6] =  6.66107313738753120669e-02; /* 0x3FB10D66, 0xA0D03D51 */
    aT[7] = -5.83357013379057348645e-02; /* 0xBFADDE2D, 0x52DEFD9A */
    aT[8] =  4.97687799461593236017e-02; /* 0x3FA97B4B, 0x24760DEB */
    aT[9] = -3.65315727442169155270e-02; /* 0xBFA2B444, 0x2C6A6C2F */
    aT[10]=  1.62858201153657823623e-02; /* 0x3F90AD3A, 0xE322DA11 */

    GET_HIGH_WORD(hx,x);
    ix = hx&0x7fffffff;
    if(ix>=0x44100000) {    /* if |x| >= 2^66 */
		u_int32_t low;
		GET_LOW_WORD(low,x);
		if(ix>0x7ff00000||
                (ix==0x7ff00000&&(low!=0)))
			return x+x;             /* NaN */
		if(hx>0) return  atanhi[3]+atanlo[3];
		else     return -atanhi[3]-atanlo[3];
    } if (ix < 0x3fdc0000) {        /* |x| < 0.4375 */
		if (ix < 0x3e200000) {      /* |x| < 2^-29 */
			if(huge+x>one) return x;        /* raise inexact */
		}
		id = -1;
    } else {
        x = __fabs(x);
        if (ix < 0x3ff30000) {          /* |x| < 1.1875 */
            if (ix < 0x3fe60000) {      /* 7/16 <=|x|<11/16 */
                id = 0; x = (2.0*x-one)/(2.0+x);
            } else {                    /* 11/16<=|x|< 19/16 */
                id = 1; x  = (x-one)/(x+one);
            }
        } else {
            if (ix < 0x40038000) {      /* |x| < 2.4375 */
                id = 2; x  = (x-1.5)/(one+1.5*x);
            } else {                    /* 2.4375 <= |x| < 2^66 */
                id = 3; x  = -1.0/x;
            }
        }}
    /* end of argument reduction */
    z = x*x;
    w = z*z;
    /* break sum from i=0 to 10 aT[i]z**(i+1) into odd and even poly */
    s1 = z*(aT[0]+w*(aT[2]+w*(aT[4]+w*(aT[6]+w*(aT[8]+w*aT[10])))));
    s2 = w*(aT[1]+w*(aT[3]+w*(aT[5]+w*(aT[7]+w*aT[9]))));
    if (id<0) return x - x*(s1+s2);
    else {
		z = atanhi[id] - ((x*(s1+s2) - atanlo[id]) - x);
		return (hx<0)? -z:z;
    }
}


double Floor(double x) {
    return __floor(x);
}                                      

/*
 * End of fastmath.c
*/
