//********************************************
// Texture.cpp
//********************************************
// class CTexture
//********************************************
// pierre.alliez@rd.francetelecom.fr
// Created : 17/12/97
// Modified : 09/10/00
//********************************************

#include "stdafx.h"
#include "math.h"
#include "Texture.h"

//////////////////////////////////////////////
// LIFE CYCLE
//////////////////////////////////////////////

//********************************************
// Constructor
//********************************************
CTexture::CTexture()
{
	m_pData = NULL;
	m_Width = 0;
	m_WidthByte32 = 0;
	m_Height = 0;
	m_Depth = 0;
}

//********************************************
// Destructor
//********************************************
CTexture::~CTexture()
{
	Free();
}

//////////////////////////////////////////////
// DATAS
//////////////////////////////////////////////

//********************************************
// Alloc
//********************************************
int CTexture::Alloc(unsigned int width,
					unsigned int height,
					unsigned int depth)
{
	Free();
	
	unsigned int BytePerPixel = (unsigned int)(depth / 8);
	unsigned int Width32 = WidthByte32(width,depth);
	
	// Only rgb and rgba modes
	ASSERT(BytePerPixel == 3);
	
	m_pData = new unsigned char [Width32 * height];
	if(m_pData == NULL)
	{
		AfxMessageBox("CTexture::Alloc : Insufisant memory");
		return 0;
	}
	
	// Set members variables
	m_Width = width;
	m_WidthByte32 = Width32;
	m_Height = height;
	m_Depth = depth;
	UpdateHeader();
	
	return 1;
}

//********************************************
// Free
//********************************************
void CTexture::Free()
{
	if(m_pData != NULL)
	{
		delete [] m_pData;
		m_pData = NULL;
	}
	m_Width = 0;
	m_Height = 0;
	m_Depth = 0;
}


//////////////////////////////////////////////
// I/O
//////////////////////////////////////////////

//********************************************
// ReadFileBMP24 (*.bmp)
//********************************************
// Read windows bmp files
// Accept only 24 bits
//********************************************
int CTexture::ReadFileBMP24(char *filename)
{
	// Cleanup
	Free();
	
	// Extension
	CString string = filename;
	string.MakeLower();
	CString extension = string.Right(4);
	ASSERT(extension == ".bmp");
	TRACE("Read bmp file...");
	
	// Check for valid bmp file
	CFile file;
	CFileException ex;
	
	// Try to open file
	if(!file.Open(filename, CFile::modeRead | CFile::typeBinary,&ex))
	{
#ifdef _DEBUG
		afxDump << "File could not be opened " << ex.m_cause << "\n";
#endif
		AfxMessageBox("Unable to open file for reading");
		return 0;
	}
	TRACE("open...");
	
	// File header
	BITMAPFILEHEADER FileHeader;
	TRY
	{
		file.Read(&FileHeader,sizeof(BITMAPFILEHEADER));
	}
	CATCH(CFileException, e)
	{
#ifdef _DEBUG
		afxDump << "Error during reading " << e->m_cause << "\n";
#endif
		AfxMessageBox("Error during reading file header");
		file.Close();
		return 0;
	}
	END_CATCH
		TRACE("file header...");
	
		/*
		TRACE("FileHeader.bfType : %d\n",FileHeader.bfType);
		TRACE("FileHeader.bfSize : %d\n",FileHeader.bfSize);
		TRACE("FileHeader.bfReserved1 : %d\n",FileHeader.bfReserved1);
		TRACE("FileHeader.bfReserved2 : %d\n",FileHeader.bfReserved2);
	TRACE("FileHeader.bfOffBits : %d\n",FileHeader.bfOffBits);*/
	
	// Is it a Windows BMP file ? (BM)
	WORD sign = ((WORD) ('M' << 8) | 'B');
	if(FileHeader.bfType != sign)
	{
		AfxMessageBox("Invalid BMP file");
		file.Close();
		return 0;
	}
	
	// Image header
	TRY
	{
		file.Read(&m_Header,sizeof(BITMAPINFOHEADER));
	}
	CATCH(CFileException, e)
	{
#ifdef _DEBUG
		afxDump << "Error during reading " << e->m_cause << "\n";
#endif
		AfxMessageBox("Error during reading image header");
		file.Close();
		return 0;
	}
	END_CATCH
		TRACE("image header...");
	
	// DEBUG
	/*
	TRACE("\n");
	TRACE("IMAGE HEADER :\n");
	TRACE("biSize : %d\n",m_Header.biSize);
	TRACE("biWidth : %d\n",m_Header.biWidth);
	TRACE("biHeight : %d\n",m_Header.biHeight);
	TRACE("biPlanes : %d\n",m_Header.biPlanes); 4
	TRACE("biBitCount : %d\n",m_Header.biBitCount); 8
	TRACE("biCompression : %d\n",m_Header.biCompression);
	TRACE("biSizeImage : %d\n",m_Header.biSizeImage);
	TRACE("biXPelsPerMeter : %d\n",m_Header.biXPelsPerMeter);
	TRACE("biYPelsPerMeter : %d\n",m_Header.biYPelsPerMeter);
	TRACE("biClrUsed : %d\n",m_Header.biClrUsed);
	TRACE("biClrImportant : %d\n",m_Header.biClrImportant);*/
	
	// 24 bits ?
	if(m_Header.biPlanes != 1 ||
		m_Header.biBitCount != 24)
	{
		AfxMessageBox("Texture file must have 24 bits depth");
		file.Close();
		return 0;
	}
	
	// Alloc (does call Free before)
	Free();
	m_pData = new unsigned char[m_Header.biSizeImage];
	if(m_pData == NULL)
	{
		AfxMessageBox("Insuffisant memory");
		file.Close();
		return 0;
	}
	
	// Update datas
	m_Width = m_Header.biWidth;
	m_Height = m_Header.biHeight;
	m_Depth = m_Header.biBitCount;
	
	// Image reading
	// Jump before CFile
	file.Seek(FileHeader.bfOffBits,CFile::begin);   
	TRY
	{
		file.Read(m_pData,m_Header.biSizeImage);
	}
	CATCH(CFileException, e)
	{
#ifdef _DEBUG
		afxDump << "Error during reading " << e->m_cause << "\n";
#endif
		AfxMessageBox("Error during reading image");
		file.Close();
		return 0;
	}
	END_CATCH
		TRACE("data (%d x %d x %d bits)...",
		m_Header.biWidth,m_Header.biHeight,m_Header.biBitCount);
	
	// Close file
	file.Close();
	TRACE("close...");
	
	UpdateWidthByte32();
	
	TRACE("done\n");
	
	return 1;
}

//********************************************
// SaveFileBMP24 (*.bmp)
//********************************************
// Save windows bmp files
// Accept only 24 bits
//********************************************
int CTexture::SaveFileBMP24(char *filename)
{
	if(m_Depth != 24)
		return 0;
	
	// Check for valid bmp file
	CFile file;
	CFileException ex;
	
	// Try to open file
	if(!file.Open(filename,CFile::modeCreate | 
		CFile::modeWrite | CFile::typeBinary,&ex))
	{
#ifdef _DEBUG
		afxDump << "File could not be opened " << ex.m_cause << "\n";
#endif
		AfxMessageBox("Unable to open file for writing");
		return 0;
	}
	
	// File header
	BITMAPFILEHEADER FileHeader;
	WORD sign = ((WORD) ('M' << 8) | 'B');
	FileHeader.bfType = sign;
	FileHeader.bfSize = 14 + 40 + m_WidthByte32*m_Height; 
	FileHeader.bfReserved1 = 0; 
	FileHeader.bfReserved2 = 0; 
	FileHeader.bfOffBits = 54; 
	
	TRACE("\nSave BMP File...\n");
	TRACE("FileHeader.bfType : %d\n",FileHeader.bfType);
	TRACE("FileHeader.bfSize : %d\n",FileHeader.bfSize);
	TRACE("FileHeader.bfReserved1 : %d\n",FileHeader.bfReserved1);
	TRACE("FileHeader.bfReserved2 : %d\n",FileHeader.bfReserved2);
	TRACE("FileHeader.bfOffBits : %d\n",FileHeader.bfOffBits);
	
	TRY
	{
		file.Write(&FileHeader,sizeof(BITMAPFILEHEADER));
	}
	CATCH(CFileException, e)
	{
#ifdef _DEBUG
		afxDump << "Error during writing " << e->m_cause << "\n";
#endif
		AfxMessageBox("Error during writing file header");
		file.Close();
		return 0;
	}
	END_CATCH
		
		// Image header
		TRY
	{
		file.Write(&m_Header,sizeof(BITMAPINFOHEADER));
	}
	CATCH(CFileException, e)
	{
#ifdef _DEBUG
		afxDump << "Error during writing " << e->m_cause << "\n";
#endif
		AfxMessageBox("Error during writing image header");
		file.Close();
		return 0;
	}
	END_CATCH
		
		// DEBUG
		TRACE("\n");
	TRACE("IMAGE HEADER :\n");
	TRACE("biSize : %d\n",m_Header.biSize);
	TRACE("biWidth : %d\n",m_Header.biWidth);
	TRACE("biHeight : %d\n",m_Header.biHeight);
	TRACE("biPlanes : %d\n",m_Header.biPlanes);
	TRACE("biBitCount : %d\n",m_Header.biBitCount);
	TRACE("biCompression : %d\n",m_Header.biCompression);
	TRACE("biSizeImage : %d\n",m_Header.biSizeImage);
	TRACE("biXPelsPerMeter : %d\n",m_Header.biXPelsPerMeter);
	TRACE("biYPelsPerMeter : %d\n",m_Header.biYPelsPerMeter);
	TRACE("biClrUsed : %d\n",m_Header.biClrUsed);
	TRACE("biClrImportant : %d\n",m_Header.biClrImportant);
	
	// Image writing
	TRY
	{
		file.Write(m_pData,m_Header.biSizeImage);
	}
	CATCH(CFileException, e)
	{
#ifdef _DEBUG
		afxDump << "Error during writing " << e->m_cause << "\n";
#endif
		AfxMessageBox("Error during writing image");
		file.Close();
		return 0;
	}
	END_CATCH
		
		// Close file
		file.Close();
	
	return 1;
}


//********************************************
// ReadFromResource
//********************************************
int CTexture::ReadFromResource(const WORD ResourceId)
{
	// Check
	ASSERT(ResourceId != 0);
	
	// Secret hand-made decomp. method
	HINSTANCE hInst = AfxGetResourceHandle();
	HRSRC hrsrc = ::FindResource(hInst,MAKEINTRESOURCE(ResourceId),"DIB");
	if(hrsrc == NULL) 
	{
		TRACE("DIB resource not found");
		return 0;
	}
	
	// Famous decomp. algorithm
	HGLOBAL hg = LoadResource(hInst, hrsrc); // you do not need to call any free
	if(hg == NULL) 
	{
		TRACE("Failed to load DIB resource");
		return FALSE;
	}
	
	void* pRes = LockResource(hg);
	ASSERT(pRes != NULL);
	int size = ::SizeofResource(hInst, hrsrc);
	TRACE("Size resource : %d\n",size);
	
	// This must be a windows .BMP file
	BITMAPFILEHEADER FileHeader;
	memcpy(&FileHeader,pRes,sizeof(BITMAPFILEHEADER));
	
	TRACE("FILE HEADER :\n");
	TRACE("FileHeader.bfType : %d\n",FileHeader.bfType);
	TRACE("FileHeader.bfSize : %d\n",FileHeader.bfSize);
	TRACE("FileHeader.bfReserved1 : %d\n",FileHeader.bfReserved1);
	TRACE("FileHeader.bfReserved2 : %d\n",FileHeader.bfReserved2);
	TRACE("FileHeader.bfOffBits : %d\n",FileHeader.bfOffBits);
	
	// Is it a Windows BMP file ? (BM)
	if(FileHeader.bfType != 0x4D42)
	{
		AfxMessageBox("Invalid BMP file");
		return 0;
	}
	TRACE("yes, this is a valid BMP image\n");
	
	// Image header
	memcpy(&m_Header,(BYTE *)pRes+sizeof(BITMAPFILEHEADER),sizeof(BITMAPINFOHEADER));
	
	// DEBUG
	TRACE("\n");
	TRACE("IMAGE HEADER :\n");
	TRACE("biSize : %d\n",m_Header.biSize);
	TRACE("biWidth : %d\n",m_Header.biWidth);
	TRACE("biHeight : %d\n",m_Header.biHeight);
	TRACE("biPlanes : %d\n",m_Header.biPlanes);
	TRACE("biBitCount : %d\n",m_Header.biBitCount);
	TRACE("biCompression : %d\n",m_Header.biCompression);
	TRACE("biSizeImage : %d\n",m_Header.biSizeImage);
	TRACE("biXPelsPerMeter : %d\n",m_Header.biXPelsPerMeter);
	TRACE("biYPelsPerMeter : %d\n",m_Header.biYPelsPerMeter);
	TRACE("biClrUsed : %d\n",m_Header.biClrUsed);
	TRACE("biClrImportant : %d\n",m_Header.biClrImportant);
	
	// 24 bits ?
	if(m_Header.biPlanes != 1 ||
		m_Header.biBitCount != 24)
	{
		AfxMessageBox("Image file must have 24 bits depth");
		return 0;
	}
	TRACE("yes, this is a 24 bits image\n");
	
	// Alloc (does call Free before)
	Free();
	TRACE("Alloc image\n");
	m_pData = new unsigned char[m_Header.biSizeImage];
	if(m_pData == NULL)
	{
		AfxMessageBox("Insuffisant memory");
		return 0;
	}
	
	// Update datas
	m_Width = m_Header.biWidth;
	m_Height = m_Header.biHeight;
	m_Depth = m_Header.biBitCount;
	TRACE("Image : (%d x %d x %d bits)\n",m_Width,m_Height,m_Depth);
	
	// Image reading
	CMemoryException e;
	memcpy(m_pData,(BYTE *)((BYTE *)pRes+FileHeader.bfOffBits),m_Header.biSizeImage);
	
	UpdateWidthByte32();
	
	return 1;
}


//********************************************
// UpdateWidthByte32
//********************************************
void CTexture::UpdateWidthByte32()
{
	m_WidthByte32 = WidthByte32(m_Width,m_Depth);
}

//********************************************
// WidthByte32
//********************************************
unsigned int CTexture::WidthByte32(unsigned int width,
								   unsigned int depth)
{
	// 32 bits alignment (4 bytes)
	int rest=(width*depth/8)%4;
	if(rest != 0)
		return (width*depth/8 + 4-rest);
	else
		return (width*depth/8);
}

//********************************************
// UpdateHeader
//********************************************
void CTexture::UpdateHeader()
{
	UpdateWidthByte32();
	
	m_Header.biWidth = m_Width;
	m_Header.biHeight = m_Height;
	m_Header.biSizeImage = m_WidthByte32 * m_Height;
	
	m_Header.biSize = 40;
	m_Header.biPlanes = 1;
	m_Header.biBitCount = m_Depth;
	m_Header.biCompression = 0;
	m_Header.biXPelsPerMeter = 0;
	m_Header.biYPelsPerMeter = 0;
	m_Header.biClrUsed = 0;
	m_Header.biClrImportant = 0;
}


//********************************************
// Flip BGR to RGB
//********************************************
int CTexture::BGRtoRGB(void)
{
	unsigned char pixel;
	int BytePerPixel = m_Depth/8;
	for(unsigned int j=0;j<m_Height;j++)
		for(unsigned int i=0;i<m_Width;i++)
		{
			pixel = m_pData[m_WidthByte32*j+i*BytePerPixel+2];
			m_pData[m_WidthByte32*j+i*BytePerPixel+2] = 
				m_pData[m_WidthByte32*j+i*BytePerPixel];
			m_pData[m_WidthByte32*j+i*BytePerPixel] = pixel;
		}
		return 1;
}


//////////////////////////////////////////////
// DISPLAY
//////////////////////////////////////////////

//********************************************
// Draw
//********************************************
int CTexture::Draw(CDC *pDC,
				   int xOffset,
				   int yOffset,
				   int width,
				   int height)
{
	// Flags	
	if(width == -1)
		width = m_Width;
	if(height == -1)
		height = m_Height;
	
	// Painting
	return SetDIBitsToDevice(pDC->m_hDC,
		                     xOffset,
							 yOffset,
							 width,
							 height,
							 0,
							 0,
							 0,
							 m_Height,
							 GetData(),
							 (CONST BITMAPINFO *)&m_Header,
							 DIB_RGB_COLORS);
}

//////////////////////////////////////////////
// CLIPBOARD
//////////////////////////////////////////////

//********************************************
// ExportHandle
//********************************************
HANDLE CTexture::ExportHandle()
{
	HANDLE handle;
	
	TRACE("Export handle...");
	
	// Process source handle size
	int size = sizeof(BITMAPINFOHEADER) + m_WidthByte32 * m_Height;
	
	// Alloc memory
	TRACE("alloc...");
	handle = (HANDLE)::GlobalAlloc (GHND,size);
	if(handle != NULL)
	{
		char *pData = (char *) ::GlobalLock((HGLOBAL)handle);
		TRACE("lock...");
		// Copy header
		TRACE("header...");
		memcpy(pData,&m_Header,sizeof(BITMAPINFOHEADER));
		// Copy datas
		TRACE("datas...");
		memcpy(pData+sizeof(BITMAPINFOHEADER),m_pData,m_WidthByte32*m_Height);
		// Unlock
		TRACE("unlock...");
		::GlobalUnlock((HGLOBAL)handle);
	}
	TRACE("ok\n");
	return handle;
}

//********************************************
// ImportHandle
//********************************************
int CTexture::ImportHandle(HANDLE handle)
{
	TRACE("Import handle...");
	ASSERT(handle != NULL);
	char *pData = (char *) ::GlobalLock((HGLOBAL)handle);
	
	// Header
	memcpy(&m_Header,pData,sizeof(BITMAPINFOHEADER));
	
	TRACE("\n");
	TRACE("IMAGE HEADER :\n");
	TRACE("biSize : %d\n",m_Header.biSize);
	TRACE("biWidth : %d\n",m_Header.biWidth);
	TRACE("biHeight : %d\n",m_Header.biHeight);
	TRACE("biPlanes : %d\n",m_Header.biPlanes);
	TRACE("biBitCount : %d\n",m_Header.biBitCount);
	TRACE("biCompression : %d\n",m_Header.biCompression);
	TRACE("biSizeImage : %d\n",m_Header.biSizeImage);
	TRACE("biXPelsPerMeter : %d\n",m_Header.biXPelsPerMeter);
	TRACE("biYPelsPerMeter : %d\n",m_Header.biYPelsPerMeter);
	TRACE("biClrUsed : %d\n",m_Header.biClrUsed);
	TRACE("biClrImportant : %d\n",m_Header.biClrImportant);
	
	// 24 bits ?
	if(m_Header.biPlanes != 1 ||
		m_Header.biBitCount != 24)
	{
		AfxMessageBox("Texture file must have 24 bits depth");
		return 0;
	}
	
	// Alloc (does call Free before)
	Alloc(m_Header.biWidth,m_Header.biHeight,m_Header.biBitCount);
	memcpy(m_pData,pData+sizeof(BITMAPINFOHEADER),m_WidthByte32*m_Height);
	
	::GlobalUnlock((HGLOBAL)handle);
	return 1;
}


//********************************************
// Insert a sub-image
// Requires a smaller image
//********************************************
int CTexture::Insert(CTexture *pImage,
					 unsigned int xOffset,
					 unsigned int yOffset)
{
	TRACE("Insert sub-image...");
	unsigned int width = pImage->GetWidth();
	unsigned int height = pImage->GetHeight();
	TRACE("(%d,%d) at (%d,%d)...",width,height,xOffset,yOffset);

	// Check sub-image is smaller
	if((width+xOffset) > m_Width ||
	   (height+yOffset) > m_Height)
	{
		TRACE("invalid sizes\n");
		return 0;
	}

	// Insert
	int BytePerPixel = m_Depth/8;
	unsigned char *pData = pImage->GetData();
	// BMP images -> lines are 32-bits aligned
	// Order : bottom->top, left->right
	unsigned int WidthByte32 = pImage->GetWidthByte32();
	for(unsigned int j=0;j<height;j++)
		for(unsigned int i=0;i<width;i++)
			for(unsigned int k=0;k<3;k++)
				m_pData[m_WidthByte32*(m_Height-1-yOffset-height+j)+
				        (i+xOffset)*BytePerPixel+k] = 
					pData[WidthByte32*j+i*BytePerPixel+k];

	TRACE("  ok\n");
	return 1;
}

//********************************************
// Insert a sub-image
// Requires a smaller image
//********************************************
int CTexture::AdaptiveInsert(CTexture *pImage,
					 unsigned int xOffset,
					 unsigned int yOffset,
					 unsigned char r,
					 unsigned char g,
					 unsigned char b)
{
	TRACE("Insert sub-image...");
	unsigned int width = pImage->GetWidth();
	unsigned int height = pImage->GetHeight();
	TRACE("(%d,%d) at (%d,%d)...",width,height,xOffset,yOffset);

	// Check sub-image is smaller
	if((width+xOffset) > m_Width ||
	   (height+yOffset) > m_Height)
	{
		TRACE("invalid sizes\n");
		return 0;
	}

	// Insert
	int BytePerPixel = m_Depth/8;
	unsigned char *pData = pImage->GetData();
	// BMP images -> lines are 32-bits aligned
	// Order : bottom->top, left->right
	unsigned int WidthByte32 = pImage->GetWidthByte32();
	for(unsigned int j=0;j<height;j++)
		for(unsigned int i=0;i<width;i++)
			for(unsigned int k=0;k<3;k++)
			{
				if(pData[WidthByte32*j+i*BytePerPixel+0] != b ||
					 pData[WidthByte32*j+i*BytePerPixel+1] != g ||
					 pData[WidthByte32*j+i*BytePerPixel+2] != r)
					m_pData[m_WidthByte32*(m_Height-1-yOffset-height+j)+
				  (i+xOffset)*BytePerPixel+k] = 
					pData[WidthByte32*j+i*BytePerPixel+k];
			}

	TRACE("  ok\n");
	return 1;
}


//*********************************
// Fill 
//*********************************
void CTexture::Fill(unsigned char red,
										unsigned char green,
										unsigned char blue)
{
	int BytePerPixel = m_Depth/8;
	for(unsigned int j=0;j<m_Height;j++)
		for(unsigned int i=0;i<m_Width;i++)
		{
			// order -> bgr
			m_pData[m_WidthByte32*j+i*BytePerPixel+0] = blue;
			m_pData[m_WidthByte32*j+i*BytePerPixel+1] = green;
			m_pData[m_WidthByte32*j+i*BytePerPixel+2] = red;
		}
}


// ** EOF **

